package ru.yandex.travel.commons.experiments;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.http.CommonHttpHeaders;


@Slf4j
public class ExperimentDataProvider {
    private Map<Class<?>, Map<String, Field>> annotatedFieldsByType = new ConcurrentHashMap<>();

    private Map<String, Field> getAnnotatedFields(Class<?> expClass) {
        Map<String, Field> annotatedFields = new HashMap<>();
        for (var field : expClass.getDeclaredFields()) {
            if (field.isAnnotationPresent(ExperimentFlag.class)) {
                ExperimentFlag ann = field.getAnnotation(ExperimentFlag.class);
                Preconditions.checkState(!annotatedFields.containsKey(ann.value()),
                        String.format("Experiment %s is already bound", ann.value()));
                annotatedFields.put(ann.value(), field);
            }
        }
        return ImmutableMap.copyOf(annotatedFields);
    }

    public <T> T getInstance(Class<T> expClass, CommonHttpHeaders headers) {
        Map<String, Field> annotatedFields = annotatedFieldsByType.computeIfAbsent(expClass, this::getAnnotatedFields);
        T instance;
        try {
            instance = expClass.getConstructor().newInstance();
        } catch (Exception e) {
            log.warn("Unable to create experimental features container for class {}", expClass, e);
            return null;
        }
        if (headers != null && headers.getExperiments() != null) {
            for (var exp : headers.getExperiments().keySet()) {
                Field f = annotatedFields.get(exp);
                if (f == null) {
                    continue;
                }
                try {
                    f.setAccessible(true);
                    if (f.getType().equals(boolean.class)) {
                        if (headers.getExperiments().get(exp).equals("enabled")) {
                            f.setBoolean(instance, true);
                        }
                    } else {
                        f.set(instance, headers.getExperiments().get(exp));
                    }
                } catch (IllegalAccessException ex) {
                    log.warn("Unable to set the value of experimental flag {}", exp, ex);
                }
            }
        }
        return instance;
    }
}
