package ru.yandex.travel.commons.http;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapLikeType;
import io.grpc.Context;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;

import ru.yandex.travel.white_label.proto.EWhiteLabelPartnerId;


@Slf4j
public class CommonHttpHeaders {
    public enum HeaderType {
        USER_GID("X-Ya-User-Gid", "Value of yandex_gid cookie", null, false),
        USER_DEVICE("X-Ya-User-Device", "User device", null, false),
        USER_LOGIN("X-Ya-Login", "Passport Login", null, false),
        USER_IP("X-Ya-User-Ip", "User IP", null, false),
        USER_TICKET("X-Ya-User-Ticket", "User Passport Ticket", null, false),
        USER_IS_STAFF("X-Ya-User-IsStaff", "Is Staff User", null, false),
        USER_IS_PLUS("X-Ya-User-IsPlus", "Is Yandex-Plus User", null, false),
        EXP_BOXES("X-Ya-ExpBoxes", "Experiment boxes", null, false),
        REQUEST_ID("X-Request-Id", "Request Id", null, false),
        ICOOKIE("X-Ya-ICookie", "ICookie", null, false),
        SESSION_KEY("X-Ya-Session-Key", "Session Key for anon user", "00000000-0000-0000-0000-000000000000", false),
        YANDEX_UID("X-Ya-YandexUid", "YandexUid", "123", true),
        PASSPORT_ID("X-Ya-PassportId", "Passport Identifier", null, false),
        SERVICE_TICKET("X-Ya-Service-Ticket", "Service Ticket", null, true),
        AWACS_REQUEST_ID("X-Ya-Awacs-Request-Id", "Balancer Request Id", null, false),
        REAL_USER_AGENT("X-Ya-User-Agent", "Real user agent", null, false),
        EXPERIMENTS("X-Ya-Uaas-experiments", "Map of active A/B experiments", null, false),
        WHITE_LABEL_PARTNER_ID("X-Ya-White-Label-Partner-Id", "Id of White Label partner", null, false),
        APP_PLATFORM("X-Ya-App-Platform", "ios, android", null, false),
        SERVICE("X-Ya-Service", "Service name", null, false);

        @Getter
        private final String header;
        @Getter
        private final String description;
        @Getter
        private final String defaultSwaggerValue;
        @Getter
        private final boolean required;

        HeaderType(String header, String description, String defaultSwaggerValue, boolean required) {
            this.header = header;
            this.description = description;
            this.defaultSwaggerValue = defaultSwaggerValue;
            this.required = required;
        }
    }

    public final static Context.Key<CommonHttpHeaders> KEY = Context.key("COMMON_HTTP_HEADERS");

    private final Map<HeaderType, String> headers;

    private final AtomicReference<Map<String, String>> experiments = new AtomicReference<>();

    public CommonHttpHeaders(Map<HeaderType, String> headers) {
        this.headers = Collections.unmodifiableMap(headers);
    }

    public static CommonHttpHeaders get() {
        return KEY.get();
    }

    public String getUserGid() {
        return headers.get(HeaderType.USER_GID);
    }

    public String getUserDevice() {
        return headers.get(HeaderType.USER_DEVICE);
    }

    public String getUserLogin() {
        return headers.get(HeaderType.USER_LOGIN);
    }

    public String getUserIP() {
        return headers.get(HeaderType.USER_IP);
    }

    public String getUserTicket() {
        return headers.get(HeaderType.USER_TICKET);
    }

    private String getUserIsStaff() {
        return headers.get(HeaderType.USER_IS_STAFF);
    }

    public boolean getUserIsStaffAsBool() {
        return Boolean.parseBoolean(getUserIsStaff());
    }

    public boolean getUserIsPlusAsBool() {
        return Boolean.parseBoolean(getUserIsPlus());
    }

    private String getUserIsPlus() {
        return headers.get(HeaderType.USER_IS_PLUS);
    }

    public String getExpBoxes() {
        return headers.get(HeaderType.EXP_BOXES);
    }

    public String getTestIdsFromExpBoxes() {
        var testIds = getTestIdsFromExpBoxesImpl();
        if (testIds != null) {
            return testIds.collect(Collectors.joining(","));
        }
        return null;
    }

    public List<Long> getIntegerTestIdsFromExpBoxes() {
        return parseIntList(getTestIdsFromExpBoxesImpl());
    }

    private Stream<String> getTestIdsFromExpBoxesImpl() {
        return getDataFromExpBoxesImpl(0);
    }

    public String getTestBucketsFromExpBoxes() {
        var buckets = getTestBucketsFromExpBoxesImpl();
        if (buckets != null) {
            return buckets.collect(Collectors.joining(","));
        }
        return null;
    }

    public List<Long> getIntegerTestBucketsFromExpBoxes() {
        return parseIntList(getTestBucketsFromExpBoxesImpl());
    }

    private Stream<String> getTestBucketsFromExpBoxesImpl() {
        return getDataFromExpBoxesImpl(2);
    }

    private List<Long> parseIntList(Stream<String> stream) {
        if (stream == null) {
            return null;
        }
        return stream.map(Long::valueOf).collect(Collectors.toUnmodifiableList());
    }

    private Stream<String> getDataFromExpBoxesImpl(int index) {
        String expBoxes = getExpBoxes();
        if (expBoxes != null) {
            return Arrays.stream(expBoxes.split(";")).map(x -> x.split(",")[index]);
        }
        return null;
    }

    public String getRequestId() {
        return headers.get(HeaderType.REQUEST_ID);
    }

    public String getICookie() {
        return headers.get(HeaderType.ICOOKIE);
    }

    public String getICookieDecrypted() {
        String encCookie = getICookie();
        if (encCookie != null) {
            return decryptICookie(encCookie);
        } else {
            return null;
        }
    }

    public native String decryptICookie(String icookie);

    public String getSessionKey() {
        return headers.get(HeaderType.SESSION_KEY);
    }

    public String getYandexUid() {
        return headers.get(HeaderType.YANDEX_UID);
    }

    public String getPassportId() {
        return headers.get(HeaderType.PASSPORT_ID);
    }

    public String getServiceTicket() {
        return headers.get(HeaderType.SERVICE_TICKET);
    }

    public String getAwacsRequestId() {
        return headers.get(HeaderType.AWACS_REQUEST_ID);
    }

    public String getRealUserAgent() {
        return headers.get(HeaderType.REAL_USER_AGENT);
    }

    public String getRawExperiments() {
        return headers.get(HeaderType.EXPERIMENTS);
    }

    public Map<String, String> getExperiments() {
        var exps = this.experiments.get();
        if (exps == null) {
            String expString = getRawExperiments();
            exps = deserializeExperiments(expString);
            experiments.compareAndSet(null, exps);
        }
        return exps;
    }

    private static Map<String, String> deserializeExperiments(String value) {
        if (Strings.isBlank(value)) {
            return Collections.emptyMap();
        }
        ObjectMapper mapper = new ObjectMapper();
        MapLikeType type = mapper.getTypeFactory().constructMapLikeType(HashMap.class, String.class, String.class);
        try {
            return mapper.readerFor(type).readValue(value);
        } catch (IOException e) {
            log.warn("Unable to load map of experiments", e);
            return Collections.emptyMap();
        }
    }

    public String getAppPlatform() {
        return headers.get(HeaderType.APP_PLATFORM);
    }

    public String getService() {
        return headers.get(HeaderType.SERVICE);
    }


    static {
        System.loadLibrary("icookie_lib_java");
    }

    public String getWhiteLabelPartnerIdStr() {
        return headers.get(HeaderType.WHITE_LABEL_PARTNER_ID);
    }

    public EWhiteLabelPartnerId getWhiteLabelPartnerId() {
        String whiteLabelPartnerId = headers.get(HeaderType.WHITE_LABEL_PARTNER_ID);
        if (whiteLabelPartnerId == null) {
            return EWhiteLabelPartnerId.WL_UNKNOWN;
        }

        switch (whiteLabelPartnerId) {
            case "s7":
                return EWhiteLabelPartnerId.WL_S7;
            case "tinkoff":
                return EWhiteLabelPartnerId.WL_TINKOFF;
            case "vtb":
                return EWhiteLabelPartnerId.WL_VTB;
            case "raiffeisen":
                return EWhiteLabelPartnerId.WL_RAIFFEISEN;
            case "tele2":
                return EWhiteLabelPartnerId.WL_TELE2;
            case "otkritie":
                return EWhiteLabelPartnerId.WL_OTKRITIE;
            case "beeline":
                return EWhiteLabelPartnerId.WL_BEELINE;
            default:
                return EWhiteLabelPartnerId.WL_UNKNOWN;
        }
    }
}
