package ru.yandex.travel.commons.http.apiclient;

import java.io.IOException;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;

@RequiredArgsConstructor
@Slf4j
public class AsyncHttpClientBase {

    @FunctionalInterface
    public interface BodySerializer<T> {
        String serialize(T bodyObject) throws Exception;
    }

    @FunctionalInterface
    public interface ResponseParser<T> {
        T parse(Response response) throws Exception;
    }

    @FunctionalInterface
    public interface ResponseBodyParser<T> {
        T parse(String s) throws Exception;
    }

    public static final Set<Integer> DEFAULT_SUCCESSFUL_RESPONSE_CODES = Set.of(200, 201);

    protected final AsyncHttpClientWrapper asyncHttpClient;
    protected final HttpApiPropertiesBase config;

    protected <RQ, RS> CompletableFuture<RS> sendRequest(
            HttpMethod method,
            String path,
            RQ bodyObject,
            BodySerializer<RQ> requestBodySerializer,
            ResponseParser<RS> responseParser,
            String purpose
    ) {
        String body;
        try {
            body = requestBodySerializer.serialize(bodyObject);
        } catch (Exception e) {
            log.error("Error serializing request", e);
            throw new RuntimeException(e);
        }
        RequestBuilder rb = createBaseRequestBuilder(method, path, body);
        return asyncHttpClient.executeRequest(rb, purpose)
                .thenApply(r -> parseResponse(
                        r,
                        responseParser,
                        this::defaultSuccessfulResponseMatcher,
                        this::defaultRetryableResponseMatcher
                ));
    }

    protected <RQ, RS> CompletableFuture<RS> sendRequest(
            HttpMethod method,
            String path,
            RQ bodyObject,
            BodySerializer<RQ> requestBodySerializer,
            ResponseBodyParser<RS> responseBodyParser,
            String purpose
    ) {
        return sendRequest(method, path, bodyObject, requestBodySerializer, (Response response) -> responseBodyParser.parse(response.getResponseBody()), purpose);
    }

    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        return new RequestBuilder()
                .setUrl(config.getBaseUrl() + path)
                .setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(config.getHttpRequestTimeout().toMillis()))
                .setMethod(method.name())
                .setBody(body);
    }

    protected <T> T parseResponse(
            Response response,
            ResponseParser<T> responseBodyParser,
            Predicate<Response> isSuccessfulFn,
            Predicate<Response> isRetryableFn
    ) {
        int sc = response.getStatusCode();
        if (isSuccessfulFn.test(response)) {
            return parseSuccessfulResponse(response, responseBodyParser);
        } else if (isRetryableFn.test(response)) {
            throw new HttpApiRetryableException("Got retryable status code: " + sc, null, sc, readErroneousContent(response));
        } else {
            throw new HttpApiException("Got response with unexpected status code: " + response.getStatusCode(),
                    null, response.getStatusCode(), readErroneousContent(response));
        }
    }

    protected boolean defaultSuccessfulResponseMatcher(Response response) {
        return DEFAULT_SUCCESSFUL_RESPONSE_CODES.contains(response.getStatusCode());
    }

    protected boolean defaultRetryableResponseMatcher(Response response) {
        return response.getStatusCode() >= 500;
    }

    protected <T> T parseSuccessfulResponse(Response response, ResponseParser<T> responseBodyParser) {
        try {
            return responseBodyParser.parse(response);
        } catch (Exception e) {
            log.error("Unable to parse response", e);
            throw new HttpApiParsingException("Unable to parse response", e, response.getStatusCode(), response.getResponseBody());
        }
    }

    protected Object readErroneousContent(Response response) {
        return response.getResponseBody();
    }

    protected <T> T sync(CompletableFuture<T> future) throws HttpApiException {
        try {
            long timeout = config.getHttpRequestTimeout().toMillis();
            return future.get(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            log.error("Http call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw ExceptionUtils.throwException(translateRpcError(e));
        } catch (ExecutionException | TimeoutException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            }
            throw ExceptionUtils.throwException(translateRpcError(e.getCause()));
        }
    }

    /**
     * Final exception transformation for {@linkplain #sync(CompletableFuture)} method users.
     * The main point is to convert all retryable exceptions here to a generic one which is supported by the caller.
     */
    protected Throwable translateRpcError(Throwable t) {
        if (t instanceof InterruptedException || t instanceof TimeoutException || t instanceof IOException) {
            return new HttpApiRetryableException(t.getMessage(), t, -500, null);
        }
        return t;
    }
}
