package ru.yandex.travel.commons.logging;

import java.util.Collections;
import java.util.Map;
import java.util.UUID;

import com.google.common.base.Strings;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import org.slf4j.MDC;

import static ru.yandex.travel.commons.logging.CommonMdcParams.MDC_ENTITY_ID;
import static ru.yandex.travel.commons.logging.CommonMdcParams.MDC_ENTITY_TYPE;

/**
 * Allows nesting MDC context without modifying the outer code's MDC.
 * <p>Usage:
 * <pre>{@code try (var ndc = nestedMdc(Map.of("EID", "123", "ETYPE", "t1"))) {
 *     // ...
 * }}</pre>
 */
@AllArgsConstructor(access = AccessLevel.PRIVATE)
public class NestedMdc implements AutoCloseable {
    private final Map<String, String> previousCtx;

    public static NestedMdc nestedMdc(Map<String, String> context) {
        NestedMdc nestedCtx = new NestedMdc(MDC.getCopyOfContextMap());
        if (context != null) {
            for (Map.Entry<String, String> ctxEntry : context.entrySet()) {
                MDC.put(ctxEntry.getKey(), ctxEntry.getValue());
            }
        }
        return nestedCtx;
    }

    public static NestedMdc empty() {
        return new NestedMdc(MDC.getCopyOfContextMap());
    }

    public static NestedMdc forEntity(LogEntity entity) {
        return forEntity(entity.getLogEntityId(), entity.getLogEntityType());
    }

    public static NestedMdc forOptionalEntity(LogEntity entity) {
        return entity != null ? forEntity(entity) : noContext();
    }

    public static NestedMdc forEntity(UUID entityId, String entityType) {
        return forEntity(entityId.toString(), entityType);
    }

    public static NestedMdc forEntityId(String entityId) {
        return forEntity(entityId, "");
    }

    public static NestedMdc forEntityId(UUID entityId) {
        return forEntityId(entityId.toString());
    }

    public static NestedMdc forOptionalEntityId(UUID entityId) {
        return entityId != null ? forEntityId(entityId) : noContext();
    }

    public static NestedMdc forEntity(String entityId, String entityType) {
        NestedMdc nestedCtx = new NestedMdc(MDC.getCopyOfContextMap());
        MDC.put(MDC_ENTITY_ID, entityId);
        if (!Strings.isNullOrEmpty(entityType)) {
            MDC.put(MDC_ENTITY_TYPE, entityType);
        } else {
            MDC.remove(MDC_ENTITY_TYPE);
        }
        return nestedCtx;
    }

    @Override
    public void close() {
        MDC.setContextMap(previousCtx != null ? previousCtx : Collections.emptyMap());
    }

    public static NestedMdc noContext() {
        return NoOpNestedMdc.INSTANCE;
    }

    private static class NoOpNestedMdc extends NestedMdc {
        public static final NoOpNestedMdc INSTANCE = new NoOpNestedMdc();

        private NoOpNestedMdc() {
            super(null);
        }

        @Override
        public void close() {
        }
    }
}
