package ru.yandex.travel.commons.logging.masking;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;

import ru.yandex.bolts.function.Function;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.logging.LoggingMarkers;

@Slf4j
@Plugin(name = "logmask", category = "Converter")
@ConverterKeys({"cm"})
public class LogMaskingConverter extends LogEventPatternConverter {
    public static final String HTTP_REQUEST_HEADERS = "request_headers";
    public static final String HTTP_RESPONSE_HEADERS = "response_headers";
    public static final List<String> HEADERS_TO_MASK = List.of("Authorization", "X-Service-Token", "X-HB-Partner-Token",
            "X-IBM-Client-Id", "X-ApiKey", "token", CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(),
            CommonHttpHeaders.HeaderType.USER_TICKET.getHeader(),
            CommonHttpHeaders.HeaderType.USER_IP.getHeader(), "X-User-Ip"
            );
    private static final String NAME = "cm";
    private static final String REMOVED = "**REMOVED**";
    private static final Map<String, Function<JsonElement, JsonElement>> BODY_OBJECT_FIELDS_TO_MASK =
            Map.of(
                    "Access", LogMaskingConverter::maskDolphinAccess,
                    "password", LogMaskingConverter::maskSingleField,
                    "token", LogMaskingConverter::maskSingleField,
                    // personal data in Trust calls
                    "user_email", LogMaskingConverter::maskSingleField,
                    "user_phone", LogMaskingConverter::maskSingleField,

                    "attachments", LogMaskingConverter::maskAttachments
            );
    private static final String HEADER_CONTENT_TYPE = "Content-Type";
    private static final String CONTENT_TYPE_PDF = "application/pdf";
    private static final String CONTENT_TYPE_OCTET_STREAM = "application/octet-stream";
    private static final String REMOVED_PDF = "**PDF**";
    private static final String REMOVED_BINARY_DATA = "**BINARY**";
    private static final Map<String, String> CONTENT_TYPE_TO_MASK = Map.of(
            CONTENT_TYPE_PDF, REMOVED_PDF,
            CONTENT_TYPE_OCTET_STREAM, REMOVED_BINARY_DATA
    );

    public static LogMaskingConverter create() {
        return new LogMaskingConverter();
    }

    LogMaskingConverter() {
        super(NAME, NAME);
    }

    private static JsonElement maskDolphinAccess(JsonElement jsonElement) {
        if (jsonElement.isJsonObject()) {
            var accessObject = jsonElement.getAsJsonObject();
            if (accessObject.has("Case") && accessObject.get("Case").getAsString().equals("LoginPass") &&
                    accessObject.has("Fields") && accessObject.get("Fields").isJsonArray()) {
                var fields = accessObject.get("Fields").getAsJsonArray();
                if (fields.size() == 2) {
                    fields.set(0, new JsonPrimitive(REMOVED));
                    fields.set(1, new JsonPrimitive(REMOVED));
                }
            }
        }
        return null;
    }

    private static JsonElement maskSingleField(JsonElement jsonElement) {
        if (jsonElement.isJsonPrimitive()) {
            return new JsonPrimitive(REMOVED);
        } else {
            return null;
        }
    }

    private static JsonElement maskAttachments(JsonElement jsonElement) {
        if (jsonElement.isJsonNull()) {
            return jsonElement;
        }
        JsonArray attachments = new JsonArray();
        if (jsonElement.isJsonArray()) {
            attachments = jsonElement.getAsJsonArray();
        } else if (jsonElement.isJsonPrimitive()) {
            attachments = JsonParser.parseString(jsonElement.getAsString()).getAsJsonArray();
        }
        for (var attachmentElm : attachments) {
            var attachment = attachmentElm.getAsJsonObject();
            if (attachment.has("mime_type") &&
                    attachment.has("data")) {
                String mimeType = attachment.get("mime_type").getAsString();
                if (CONTENT_TYPE_TO_MASK.containsKey(mimeType)) {
                    attachment.remove("data");
                    attachment.addProperty("data", CONTENT_TYPE_TO_MASK.get(mimeType));
                }
            }
        }
        return attachments;
    }

    // required by log4j
    @SuppressWarnings("unused")
    public static LogMaskingConverter newInstance(final String[] options) {
        return new LogMaskingConverter();
    }

    public static ObjectMapper getObjectMapperForLogEvents() {
        return new ObjectMapper()
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
                .setAnnotationIntrospector(new MaskingJacksonAnnotationIntrospector())
                .registerModule(new JavaTimeModule());
    }

    private static String getIfMaskedContentType(JsonObject headersObject) {
        if (headersObject.has(HEADER_CONTENT_TYPE)) {
            String contentType = headersObject.get(HEADER_CONTENT_TYPE).getAsString();
            if (CONTENT_TYPE_TO_MASK.containsKey(contentType)) {
                return contentType;
            }
        }
        return null;
    }

    @Override
    public void format(LogEvent event, StringBuilder toAppendTo) {
        String message = event.getMessage().getFormattedMessage();
        if (event.getMarker() != null && event.getMarker().getName().equals(LoggingMarkers.HTTP_REQUEST_RESPONSE_MARKER.getName())) {
            String maskedContentType = null;
            JsonObject messageObject = JsonParser.parseString(message).getAsJsonObject();
            for (String key : List.of(HTTP_REQUEST_HEADERS, HTTP_RESPONSE_HEADERS)) {
                if (messageObject.has(key)) {
                    JsonElement headersElement = messageObject.get(key);
                    if (headersElement == null || !headersElement.isJsonObject()){
                        continue;
                    }
                    JsonObject headersObject = headersElement.getAsJsonObject();
                    maskHttpHeaders(headersObject);
                    if (maskedContentType == null) {
                        maskedContentType = getIfMaskedContentType(headersObject);
                    }
                }
            }

            for (String key : List.of("request_body", "response_body")) {
                JsonElement bodyElem = messageObject.get(key);
                if (bodyElem != null) {
                    maskBody(maskedContentType, messageObject, key, bodyElem);
                }
            }
            toAppendTo.append(messageObject);
        } else {
            toAppendTo.append(message);
        }
    }

    private void maskHttpHeaders(JsonObject headersObject) {
        for (String header : HEADERS_TO_MASK) {
            if (headersObject.has(header)) {
                headersObject.addProperty(header, REMOVED);
            }
        }
    }

    private void maskBody(String maskedContentType, JsonObject messageObject, String key, JsonElement bodyElem) {
        if (bodyElem.isJsonPrimitive() && maskedContentType != null) {
            messageObject.remove(key);
            messageObject.addProperty(key, CONTENT_TYPE_TO_MASK.get(maskedContentType));
        } else if (bodyElem.isJsonObject()) {
            try {
                JsonObject bodyObject = bodyElem.getAsJsonObject();
                for (var maskEntry : BODY_OBJECT_FIELDS_TO_MASK.entrySet()) {
                    if (bodyObject.has(maskEntry.getKey())) {
                        var value = bodyObject.get(maskEntry.getKey());
                        var replacement = maskEntry.getValue().apply(value);
                        if (replacement != null) {
                            bodyObject.remove(maskEntry.getKey());
                            bodyObject.add(maskEntry.getKey(), replacement);
                        }
                    }
                }
            } catch (Exception ex) {
                log.error("Exception happened while masking log", ex);
            }
        } else if (bodyElem.isJsonArray()) {
            // POST www-form-urlencoded case
            maskFormSubmitBody(bodyElem.getAsJsonArray());
        }
    }

    private void maskFormSubmitBody(JsonArray array) {
        for (JsonElement param : array) {
            if (!param.isJsonObject()) {
                continue;
            }
            JsonObject paramObject = param.getAsJsonObject();
            if (paramObject.has("name") && paramObject.has("value")) {
                String name = paramObject.get("name").getAsString();
                Optional.ofNullable(BODY_OBJECT_FIELDS_TO_MASK.get(name))
                        .map(fn -> fn.apply(paramObject.get("value")))
                        .ifPresent(replacement -> {
                            paramObject.remove("value");
                            paramObject.add("value", replacement);
                        });
            }
        }
    }

}
