package ru.yandex.travel.commons.messaging;

import java.io.ByteArrayOutputStream;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

import com.github.luben.zstd.Zstd;

import ru.yandex.travel.commons.proto.EMessageCodec;

public class CompressionUtils {
    public static byte[] compress(CompressionSettings compressionSettings, byte[] bytes) {
        switch (compressionSettings.getMessageCodec()) {
            case MC_NONE:
                return bytes;
            case MC_ZLIB:
                return compressZLib(compressionSettings.getCompressionLevel(), 1024, bytes);
            case MC_ZSTD:
                return compressZStd(compressionSettings.getCompressionLevel(), bytes);
            default:
                throw new UnsupportedOperationException("Incorrect EMessageCodec was specified");
        }
    }

    public static byte[] decompress(CompressionSettings compressionSettings, byte[] bytes) {
        switch (compressionSettings.getMessageCodec()) {
            case MC_NONE:
                return bytes;
            case MC_ZLIB:
                return decompressZLib(1024, bytes);
            case MC_ZSTD:
                return decompressZStd(bytes);
            default:
                throw new UnsupportedOperationException("Incorrect EMessageCodec was specified");
        }
    }

    public static byte[] decompress(EMessageCodec codec, byte[] bytes) {
        switch (codec) {
            case MC_NONE:
                return bytes;
            case MC_ZLIB:
                return decompressZLib(1024, bytes);
            case MC_ZSTD:
                return decompressZStd(bytes);
            default:
                throw new UnsupportedOperationException("Incorrect EMessageCodec was specified");
        }
    }

    public static byte[] compressZLib(int compressionLevel, int bufferSize, byte[] bytes) {
        Deflater compresser = new Deflater(compressionLevel);
        compresser.setInput(bytes);
        compresser.finish();
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream(bytes.length);
        byte[] buffer = new byte[bufferSize];
        while (!compresser.finished()) {
            int count = compresser.deflate(buffer);
            outputStream.write(buffer, 0, count);
        }
        // Explicitly call end() method to avoid OOM error (http://www.devguli.com/blog/eng/java-deflater-and-outofmemoryerror/)
        compresser.end();
        return outputStream.toByteArray();
    }

    public static byte[] decompressZLib(int bufferSize, byte[] bytes) {
        Inflater decompressor = new Inflater();
        decompressor.setInput(bytes);
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        byte[] buffer = new byte[bufferSize];
        while (!decompressor.needsInput()) {
            int count = 0;
            try {
                count = decompressor.inflate(buffer);
            } catch (DataFormatException e) {
                throw new RuntimeException("Unable to decompress the message");
            }
            outputStream.write(buffer, 0, count);
        }
        decompressor.end();
        return outputStream.toByteArray();

    }

    public static byte[] compressZStd(int compressionLevel, byte[] bytes) {
        return Zstd.compress(bytes, compressionLevel);
    }

    public static byte[] decompressZStd(byte[] bytes) {
        long size = Zstd.decompressedSize(bytes);
        byte[] dest = new byte[(int) size];
        Zstd.decompress(dest, bytes);
        return dest;
    }
}
