package ru.yandex.travel.commons.messaging;

import java.time.Duration;
import java.util.UUID;

import com.google.common.hash.Hashing;
import com.google.protobuf.Message;

import ru.yandex.travel.commons.proto.EMessageCodec;


public class Envelope {
    private final long timestamp;
    private final String messageType;
    private final EMessageCodec codec;
    private final byte[] bytes;
    private final String bytesHash;
    private final String messageId;
    private final Long expireTimestamp;

    private Envelope(long timestamp, String messageType, CompressionSettings compressionSettings, byte[] bytes,
                     String messageId, Long expireTimestamp) {
        this.timestamp = timestamp;
        this.messageType = messageType;
        this.codec = compressionSettings.getMessageCodec();
        this.bytes = CompressionUtils.compress(compressionSettings, bytes);
        this.bytesHash = Hashing.farmHashFingerprint64().hashBytes(this.bytes).toString();
        if (messageId == null) {
            this.messageId = UUID.randomUUID().toString();
        } else {
            this.messageId = messageId;
        }
        this.expireTimestamp = expireTimestamp;
    }

    public static Envelope create(Message message, CompressionSettings compressionSettings, String id, Duration lifetime) {
        long now = System.currentTimeMillis();
        Long expireTimestamp = lifetime == null ? null : now + lifetime.toMillis();
        return new Envelope(
                now ,
                message.getDescriptorForType().getFullName(),
                compressionSettings,
                message.toByteArray(),
                id,
                expireTimestamp);
    }

    public long getTimestamp() {
        return timestamp;
    }

    public String getMessageType() {
        return messageType;
    }

    public EMessageCodec getCodec() {
        return codec;
    }

    public byte[] getBytes() {
        return bytes;
    }

    public String getMessageId() {
        return messageId;
    }

    public Long getExpireTimestamp() {
        return expireTimestamp;
    }

    @Override
    public String toString() {
        return String.format("{timestamp=%d, messageType='%s', bytesHash=%s, messageId=%s, expireTimestamp=%s}",
                timestamp, messageType, bytesHash, messageId, expireTimestamp);
    }
}
