package ru.yandex.travel.commons.proto;

public class Error {
    private TError.Builder protoBuilder;

    private Error(EErrorCode code, String message) {
        protoBuilder = TError.newBuilder();
        protoBuilder.setCode(code);
        protoBuilder.setMessage(message);
    }

    private Error(EErrorCode code, String format, Object... args) {
        protoBuilder = TError.newBuilder();
        protoBuilder.setCode(code);
        protoBuilder.setMessage(format(format, args));
    }

    public Error withAttribute(String key, Object value) {
        protoBuilder.putAttribute(key, value.toString());
        return this;
    }

    public Error withCause(Throwable t) {
        // TODO(sandello): Somehow configure the verbosity here.
        protoBuilder.addNestedError(ProtoUtils.errorFromThrowable(t, false));
        return this;
    }

    public ErrorException toEx() {
        return new ErrorException(protoBuilder);
    }

    public void andThrow() {
        throw toEx();
    }

    private static class CtorClosure {
        EErrorCode code;

        CtorClosure(EErrorCode code) {
            this.code = code;
        }

        Error with(String message) {
            return new Error(code, message);
        }

        Error with(String message, Object... args) {
            return new Error(code, message, args);
        }
    }

    public static CtorClosure INVALID_ARGUMENT = new CtorClosure(EErrorCode.EC_INVALID_ARGUMENT);
    public static CtorClosure FAILED_PRECONDITION = new CtorClosure(EErrorCode.EC_FAILED_PRECONDITION);

    public static Error with(EErrorCode code, String message) {
        return new Error(code, message);
    }

    public static Error with(EErrorCode code, String format, Object... args) {
        return new Error(code, format, args);
    }

    public static void checkArgument(boolean expression, String message) {
        if (!expression) {
            throw Error.INVALID_ARGUMENT.with(message).toEx();
        }
    }

    public static void checkArgument(boolean expression, String format, Object... args) {
        if (!expression) {
            throw Error.INVALID_ARGUMENT.with(format, args).toEx();
        }
    }

    public static void checkState(boolean expression, String message) {
        if (!expression) {
            throw Error.FAILED_PRECONDITION.with(message).toEx();
        }
    }

    public static void checkState(boolean expression, String format, Object... args) {
        if (!expression) {
            throw Error.FAILED_PRECONDITION.with(format, args).toEx();
        }
    }

    // This method was copied from
    // #com.google.common.base.Preconditions.format
    // #com.google.common.base.Verify.format
    // This implementation with the string formatting does not throw in case with misaligned format string and the arguments.
    private static String format(String template, Object... args) {
        template = String.valueOf(template); // null -> "null"

        // start substituting the arguments into the '%s' placeholders
        StringBuilder builder = new StringBuilder(template.length() + 16 * args.length);
        int templateStart = 0;
        int i = 0;
        while (i < args.length) {
            int placeholderStart = template.indexOf("%s", templateStart);
            if (placeholderStart == -1) {
                break;
            }
            builder.append(template, templateStart, placeholderStart);
            builder.append(args[i++]);
            templateStart = placeholderStart + 2;
        }
        builder.append(template, templateStart, template.length());

        // if we run out with placeholders, append the extra args in square braces
        if (i < args.length) {
            builder.append(" [");
            builder.append(args[i++]);
            while (i < args.length) {
                builder.append(", ");
                builder.append(args[i++]);
            }
            builder.append(']');
        }

        return builder.toString();
    }
}
