package ru.yandex.travel.commons.proto;

import java.util.Objects;
import java.util.Set;

import javax.money.CurrencyContext;
import javax.money.CurrencyContextBuilder;
import javax.money.CurrencyUnit;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import static java.util.stream.Collectors.toMap;

public class ProtoCurrencyUnit implements CurrencyUnit {
    public static final ProtoCurrencyUnit RUB = new ProtoCurrencyUnit(ECurrency.C_RUB, "RUB", 2);
    public static final ProtoCurrencyUnit USD = new ProtoCurrencyUnit(ECurrency.C_USD, "USD", 2);
    public static final ProtoCurrencyUnit EUR = new ProtoCurrencyUnit(ECurrency.C_EUR, "EUR", 2);
    public static final ProtoCurrencyUnit JPY = new ProtoCurrencyUnit(ECurrency.C_JPY, "JPY", 2);
    // XXX is used internally by FastMoney for MAX_VALUE constant and is initialized statically
    // without XXX currency everything using FastMoney will fail with UnknownCurrencyException
    public static final ProtoCurrencyUnit XXX = new ProtoCurrencyUnit(ECurrency.C_UNKNOWN, "XXX", 0);

    public static final Set<ProtoCurrencyUnit> ALL_CURRENCY_UNITS = ImmutableSet.of(RUB, USD, EUR, JPY);

    private static final ImmutableMap<String, ProtoCurrencyUnit> BY_CURRENCY_CODE =
            ImmutableMap.copyOf(ALL_CURRENCY_UNITS.stream().collect(toMap(u -> u.currencyCode, u -> u)));

    private static final ImmutableMap<ECurrency, ProtoCurrencyUnit> BY_PROTO_CURRENCY =
            ImmutableMap.copyOf(ALL_CURRENCY_UNITS.stream().collect(toMap(u -> u.protoCurrency, u -> u)));


    private final ECurrency protoCurrency;
    private final int precision;
    @JsonValue
    private final String currencyCode;
    private final CurrencyContext currencyContext;

    private ProtoCurrencyUnit(ECurrency protoCurrency, String currencyCode, int precision) {
        this.protoCurrency = protoCurrency;
        this.precision = precision;
        this.currencyCode = currencyCode;
        this.currencyContext = CurrencyContextBuilder.of(TravelProtoCurrencyUnitProvider.PROVIDER_NAME).build();
    }

    public static ProtoCurrencyUnit fromProtoCurrencyUnit(ECurrency currency) {
        ProtoCurrencyUnit result = BY_PROTO_CURRENCY.get(currency);
        if (result == null) {
            throw new IllegalArgumentException(String.format("Not registered currency: %s", currency));
        }
        return result;
    }

    @JsonCreator
    public static ProtoCurrencyUnit fromCurrencyCode(String currencyCode) {
        ProtoCurrencyUnit result = BY_CURRENCY_CODE.get(currencyCode);
        if (result == null) {
            throw new IllegalArgumentException(String.format("Not registered currency with code %s", currencyCode));
        }
        return result;
    }

    public String getCurrencyCode() {
        return currencyCode;
    }

    public ECurrency getProtoCurrency() {
        return protoCurrency;
    }

    @Override
    public int getNumericCode() {
        return -1;
    }

    @Override
    public int getDefaultFractionDigits() {
        return precision;
    }

    @Override
    public CurrencyContext getContext() {
        return currencyContext;
    }

    @Override
    public int compareTo(CurrencyUnit o) {
        return currencyCode.compareTo(o.getCurrencyCode());
    }

    /**
     * Copied from BuildableCurrencyUnit.hashCode
     */
    @Override
    public int hashCode() {
        return Objects.hashCode(currencyCode);
    }

    /**
     * Copied from BuildableCurrencyUnit.equals
     */
    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof CurrencyUnit) {
            CurrencyUnit other = (CurrencyUnit) obj;
            return Objects.equals(getCurrencyCode(), other.getCurrencyCode());
        }
        return false;
    }

    @Override
    public String toString() {
        return "ProtoCurrencyUnit(currencyCode=" + currencyCode
                + ", context=" + this.currencyContext
                + ')';
    }
}
