package ru.yandex.travel.commons.proto;

import java.io.IOException;
import java.lang.reflect.UndeclaredThrowableException;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.UUID;
import java.util.concurrent.CompletionException;

import javax.annotation.Nullable;
import javax.money.CurrencyUnit;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.Message;
import com.google.protobuf.Timestamp;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.jackson.MoneySerializersModule;

public class ProtoUtils {
    private static final ObjectMapper objectMapper = createObjectMapper();

    public static TError errorFromThrowable(Throwable t, boolean verbose) {
        // Drop extra wrappers.
        while (t instanceof CompletionException || t instanceof UndeclaredThrowableException) {
            if (t.getCause() == null) {
                break;
            }
            t = t.getCause();
        }

        // Unwrap the error.
        if (t instanceof ErrorException) {
            return ((ErrorException) t).getError();
        }

        String message = t.getMessage() != null ? t.getMessage() : t.getClass().getName();
        TError.Builder builder = TError.newBuilder()
                .setCode(EErrorCode.EC_GENERAL_ERROR)
                .setMessage(message);
        if (t.getCause() != null) {
            builder.addNestedError(errorFromThrowable(t.getCause(), verbose));
        }
        if (verbose) {
            for (Throwable t1 : t.getSuppressed()) {
                builder.addNestedError(errorFromThrowable(t1, verbose));
            }
            builder.putAttribute("class", t.getClass().getName());
            StackTraceElement[] elements = t.getStackTrace();
            if (elements.length > 0) {
                StringBuilder sb = new StringBuilder();
                for (StackTraceElement element : elements) {
                    sb.append(element.toString());
                    sb.append('\n');
                }
                builder.putAttribute("stack_trace", sb.toString());
            }
        }
        return builder.build();
    }

    private static TError.Builder validationErrorB(String message) {
        return TError.newBuilder()
                .setCode(EErrorCode.EC_INVALID_ARGUMENT)
                .setMessage(message);
    }

    public static TError validationError(String message) {
        return validationErrorB(message)
                .build();
    }

    public static TError validationError(String message, String k1, String v1) {
        return validationErrorB(message)
                .putAttribute(k1, v1)
                .build();
    }

    public static Timestamp timestamp() {
        return timestamp(System.currentTimeMillis());
    }

    public static Timestamp timestamp(long millis) {
        return Timestamp.newBuilder()
                .setSeconds(millis / 1000)
                .setNanos((int) ((millis % 1000) * 1000000))
                .build();
    }

    public static Timestamp fromLocalDateTime(LocalDateTime localDateTime) {
        return fromInstant(localDateTime.toInstant(ZoneOffset.UTC));
    }

    public static Timestamp fromLocalDate(LocalDate localDate) {
        return fromInstant(localDate.atStartOfDay().toInstant(ZoneOffset.UTC));
    }

    public static TDate toTDate(LocalDate localDate) {
        return TDate.newBuilder().setYear(localDate.getYear()).setMonth(localDate.getMonthValue())
                .setDay(localDate.getDayOfMonth()).build();
    }

    public static Timestamp fromInstant(Instant instant) {
        return Timestamp.newBuilder()
                .setSeconds(instant.getEpochSecond())
                .setNanos(instant.getNano())
                .build();
    }

    public static Timestamp fromInstantSafe(Instant instant) {
        return instant == null ? Timestamp.getDefaultInstance() : fromInstant(instant);
    }

    public static Instant toInstant(Timestamp timestamp) {
        return Instant.ofEpochSecond(timestamp.getSeconds(), timestamp.getNanos());
    }

    public static Instant toInstantFromSafe(Timestamp timestamp) {
        if (timestamp == null || timestamp.equals(Timestamp.getDefaultInstance())) {
            return null;
        }
        return Instant.ofEpochSecond(timestamp.getSeconds(), timestamp.getNanos());
    }

    public static LocalDate toLocalDate(Timestamp timestamp) {
        return toLocalDateTime(timestamp).toLocalDate();
    }

    public static LocalDate toLocalDate(TDate date) {
        return LocalDate.of(date.getYear(), date.getMonth(), date.getDay());
    }

    @Nullable
    public static LocalDate toLocalDateNullable(@Nullable TDate date) {
        if (TDate.getDefaultInstance().equals(date) || date == null) {
            return null;
        }
        return LocalDate.of(date.getYear(), date.getMonth(), date.getDay());
    }

    public static LocalDateTime toLocalDateTime(Timestamp timestamp) {
        return LocalDateTime.ofInstant(toInstant(timestamp), ZoneId.of("UTC"));
    }

    public static JsonNode fromTJson(TJson tJson) {
        try {
            return objectMapper.readTree(tJson.getValue());
        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to parse JSON", e);
        }
    }

    public static <T> T fromTJson(TJson tJson, Class<T> objectClass) {

        try {
            return objectMapper.readerFor(objectClass).readValue(tJson.getValue());
        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to parse JSON", e);
        }
    }

    public static TJson toTJson(Object object) {
        try {
            String value = objectMapper.writeValueAsString(object);
            return TJson.newBuilder()
                    .setValue(value)
                    .build();
        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to generate JSON", e);
        }
    }

    public static TJson toTJson(Object object, String version) {
        try {
            String value = objectMapper.writeValueAsString(object);
            return TJson.newBuilder()
                    .setValue(value)
                    .setVersion(version)
                    .build();
        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to generate JSON", e);
        }
    }

    public static String randomId() {
        return UUID.randomUUID().toString();
    }

    public static ObjectMapper createObjectMapper() {
        return new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .registerModule(new JavaTimeModule())
                .registerModule(new Jdk8Module())
                .registerModule(new MoneySerializersModule());
    }

    public static Money fromTPrice(TPrice tPrice) {
        CurrencyUnit currencyUnit = ProtoCurrencyUnit.fromProtoCurrencyUnit(tPrice.getCurrency());
        return Money.of(BigDecimal.valueOf(tPrice.getAmount(), tPrice.getPrecision()), currencyUnit);
    }

    public static TPrice toTPrice(Money money) {
        return toTPrice(money, null);
    }

    public static TPrice toTPrice(Money money, Integer precisionNullable) {
        Preconditions.checkArgument(money.getCurrency() instanceof ProtoCurrencyUnit, "Provided fast money instance " +
                "must have ProtoCurrencyUnit as it's currency unit");
        ProtoCurrencyUnit currencyUnit = (ProtoCurrencyUnit) money.getCurrency();
        int precision = precisionNullable != null ? precisionNullable : currencyUnit.getDefaultFractionDigits();
        return TPrice.newBuilder()
                .setAmount(money.multiply(BigDecimal.valueOf(10).pow(precision)).getNumber().longValue())
                .setCurrency(currencyUnit.getProtoCurrency())
                .setPrecision(precision)
                .build();
    }

    public static String toStringOrEmpty(UUID uuid) {
        if (uuid == null) {
            return "";
        }
        return uuid.toString();
    }

    public static UUID fromStringOrNull(String protoUuid) {
        if (Strings.isNullOrEmpty(protoUuid)) {
            return null;
        }
        return UUID.fromString(protoUuid);
    }

    public static <T extends Message> boolean hasFields(T message) {
        for (var field : message.getDescriptorForType().getFields()) {
            if (field.isRepeated() ? message.getRepeatedFieldCount(field) > 0 : message.hasField(field)) {
                return true;
            }
        }
        return false;
    }
}
