package ru.yandex.travel.commons.proto;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.money.CurrencyQuery;
import javax.money.CurrencyUnit;
import javax.money.spi.CurrencyProviderSpi;

import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;

/**
 * Our explicit CurrencyUnitProvider for the Moneta framework.
 * Most of the implementation is based on {@linkplain org.javamoney.moneta.internal.ConfigurableCurrencyUnitProvider}.
 * It's functionality is limited to currency code lookups only.
 */
@Slf4j
public class TravelProtoCurrencyUnitProvider implements CurrencyProviderSpi {
    public static final String PROVIDER_NAME = "TravelProtoCurrencyUnitProvider";

    private final ImmutableMap<String, CurrencyUnit> currencyUnits;

    public TravelProtoCurrencyUnitProvider() {
        log.info("Registering {}", PROVIDER_NAME);
        Map<String, CurrencyUnit> currencyUnits = new HashMap<>();
        for (ProtoCurrencyUnit currencyUnit : ProtoCurrencyUnit.ALL_CURRENCY_UNITS) {
            registerCurrencyUnit(currencyUnits, currencyUnit);
        }
        // special internal unit
        registerCurrencyUnit(currencyUnits, ProtoCurrencyUnit.XXX);
        this.currencyUnits = ImmutableMap.copyOf(currencyUnits);
    }

    private static void registerCurrencyUnit(Map<String, CurrencyUnit> currencyUnits, CurrencyUnit currencyUnit) {
        Objects.requireNonNull(currencyUnit);
        currencyUnits.put(currencyUnit.getCurrencyCode(), currencyUnit);
    }

    @Override
    public String getProviderName() {
        return PROVIDER_NAME;
    }

    @Override
    public Set<CurrencyUnit> getCurrencies(CurrencyQuery currencyQuery) {
        Set<CurrencyUnit> result = new HashSet<>(currencyUnits.size());
        if (currencyQuery.get(LocalDateTime.class) != null || currencyQuery.get(LocalDate.class) != null) {
            return Collections.emptySet();
        }
        if (!currencyQuery.getCurrencyCodes().isEmpty()) {
            for (String code : currencyQuery.getCurrencyCodes()) {
                CurrencyUnit cu = currencyUnits.get(code);
                if (cu != null) {
                    result.add(cu);
                }
            }
            return result;
        }
        if (!currencyQuery.getCountries().isEmpty()) {
            // country lookups aren't supported
            return result;
        }
        result.addAll(currencyUnits.values());
        return result;
    }

    @Override
    public String toString() {
        return "TravelProtoCurrencyUnitProvider [currencyUnits=" + currencyUnits + ']';
    }
}
