package ru.yandex.travel.commons.retry;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;

public class RetryStrategyBuilder<T> {
    private final Set<Class<? extends Exception>> exceptionClassesToRetry = new HashSet<>();
    private final List<Consumer<T>> resultCheckers = new ArrayList<>();
    private final List<Function<Exception, Boolean>> exceptionCheckers = new ArrayList<>();
    private int numRetries = 0;
    private Duration timeoutDuration = Duration.ZERO;
    private RetryTimeoutFunction<T> timeoutFunction = null;

    public RetryStrategyBuilder<T> retryOnExceptionClass(Class<? extends Exception> exception) {
        exceptionClassesToRetry.add(exception);
        return this;
    }

    public RetryStrategyBuilder<T> retryOnException(Function<Exception, Boolean> checkFunction) {
        exceptionCheckers.add(checkFunction);
        return this;
    }

    public RetryStrategyBuilder<T> validateResult(Consumer<T> checkFunction) {
        resultCheckers.add(checkFunction);
        return this;
    }

    public RetryStrategyBuilder<T> setTimeout(Duration timeoutDuration) {
        this.timeoutDuration = timeoutDuration;
        return this;
    }

    public RetryStrategyBuilder<T> setTimeout(long timeoutMillis) {
        this.timeoutDuration = Duration.ofMillis(timeoutMillis);
        return this;
    }

    public RetryStrategyBuilder<T> setTimeout(RetryTimeoutFunction<T> timeoutFunction) {
        this.timeoutFunction = timeoutFunction;
        return this;
    }

    public RetryStrategyBuilder<T> setNumRetries(int numRetries) {
        this.numRetries = numRetries;
        return this;
    }

    public RetryStrategy<T> build() {
        final Set<Class<? extends Exception>> exceptionClassesCopy = new HashSet<>(exceptionClassesToRetry);
        final List<Consumer<T>> resultCheckersCopy = new ArrayList<>(resultCheckers);
        final List<Function<Exception, Boolean>> exceptionCheckersCopy = new ArrayList<>(exceptionCheckers);
        final RetryTimeoutFunction<T> timeoutFunctionCopy = timeoutFunction;
        final Duration timeoutDurationCopy = timeoutDuration;
        final int numRetriesCopy = numRetries;
        return new RetryStrategy<T>() {
            @Override
            public boolean shouldRetryOnException(Exception ex) {
                if (exceptionClassesCopy.contains(ex.getClass())) {
                    return true;
                }
                if (exceptionCheckersCopy.stream().anyMatch(checker -> checker.apply(ex))) {
                    return true;
                }
                return false;
            }

            @Override
            public void validateResult(T result) {
                for (Consumer<T> f: resultCheckersCopy) {
                    f.accept(result);
                }
            }

            @Override
            public Duration getWaitDuration(int iteration, Exception ex, T result) {
                if (timeoutFunctionCopy != null) {
                    return timeoutFunctionCopy.apply(iteration, ex, result);
                } else {
                    return timeoutDurationCopy;
                }
            }

            @Override
            public int getNumRetries() {
                return numRetriesCopy;
            }
        };
    }
}
