package ru.yandex.travel.commons.text;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;

import ru.yandex.misc.lang.StringUtils;

import static java.util.Map.entry;
import static java.util.stream.Collectors.toMap;

public class Transliterator {
    private static final Map<Character, String> TRANSLIT_TABLE = buildTranslitTable();
    private static final String LATIN = "abcdefghijklmonpqrstuvwxyz";
    private static final String OTHER_ALLOWED = " -";
    private static final Set<Character> ALLOWED_CHARACTERS = buildAllowedCharacters();

    private static Map<Character, String> buildTranslitTable() {
        var lowerCaseLettersMap = Map.ofEntries(
                entry('а', "a"),
                entry('б', "b"),
                entry('в', "v"),
                entry('г', "g"),
                entry('д', "d"),
                entry('е', "e"),
                entry('ё', "e"),
                entry('ж', "zh"),
                entry('з', "z"),
                entry('и', "i"),
                entry('й', "i"),
                entry('к', "k"),
                entry('л', "l"),
                entry('м', "m"),
                entry('н', "n"),
                entry('о', "o"),
                entry('п', "p"),
                entry('р', "r"),
                entry('с', "s"),
                entry('т', "t"),
                entry('у', "u"),
                entry('ф', "f"),
                entry('х', "kh"),
                entry('ц', "ts"),
                entry('ч', "ch"),
                entry('ш', "sh"),
                entry('щ', "shch"),
                entry('ы', "y"),
                entry('ь', ""),
                entry('ъ', "ie"),
                entry('э', "e"),
                entry('ю', "iu"),
                entry('я', "ia")
        );

        var upperCaseLettersMap = lowerCaseLettersMap.entrySet().stream()
                .collect(toMap(
                        e -> Character.toUpperCase(e.getKey()),
                        e -> StringUtils.capitalize(e.getValue())
                ));

        Map<Character, String> resultMap = new HashMap<>();
        resultMap.putAll(lowerCaseLettersMap);
        resultMap.putAll(upperCaseLettersMap);
        return Map.copyOf(resultMap);
    }

    private static Set<Character> buildAllowedCharacters() {
        Set<Character> latinLowerCase =
                LATIN.chars().mapToObj(code -> Character.toLowerCase((char) code)).collect(Collectors.toSet());
        Set<Character> latinUpperCase =
                LATIN.chars().mapToObj(code -> Character.toUpperCase((char) code)).collect(Collectors.toSet());
        Set<Character> others = OTHER_ALLOWED.chars().mapToObj(code -> (char) code).collect(Collectors.toSet());
        return Sets.union(Sets.union(latinLowerCase, latinUpperCase), others);
    }

    public static String transliterate(String input) {
        if (Objects.isNull(input)) {
            return null;
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (char ch : input.toCharArray()) {
            if (TRANSLIT_TABLE.containsKey(ch)) {
                stringBuilder.append(TRANSLIT_TABLE.get(ch));
            } else if (ALLOWED_CHARACTERS.contains(ch)) {
                stringBuilder.append(ch);
            }
        }
        return stringBuilder.toString();
    }
}
