package ru.yandex.travel.commons.yt;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

import io.netty.channel.nio.NioEventLoopGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yt.ytclient.bus.BusConnector;
import ru.yandex.yt.ytclient.bus.DefaultBusConnector;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.rpc.AlwaysSwitchRpcFailoverPolicy;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;
import ru.yandex.yt.ytclient.rpc.RpcOptions;

public class ConnectionFactory {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionFactory.class);

    private BaseYtProperties config;
    private Map<String, YtClient> clientMap = new ConcurrentHashMap<>();

    private final Object busConnectorSyncRoot = new Object();
    private BusConnector busConnector = null;

    private final AtomicBoolean isClosed = new AtomicBoolean(false);

    public ConnectionFactory(BaseYtProperties config) {
        this.config = config;
    }

    public BaseYtProperties getConfig() {
        return config;
    }

    public YtClient getClientForCluster(String clusterName) {
        return clientMap.computeIfAbsent(clusterName, key -> {
            YtClusterPropertiesInterface clusterConfig = config.getClusterConfigFor(clusterName);
            logger.info("Creating YT client for cluster '{}'", key);
            RpcOptions rpcOptions = new RpcOptions();
            rpcOptions.setGlobalTimeout(clusterConfig.getGlobalTimeout())
                    .setFailoverTimeout(clusterConfig.getFailoverTimeout())
                    .setPingTimeout(clusterConfig.getPingTimeout());
            if (clusterConfig.isSwitchOnErrors()) {
                rpcOptions.setFailoverPolicy(new AlwaysSwitchRpcFailoverPolicy());
            }
            return new YtClient(
                    getBusConnector(),
                    clusterName,
                    new RpcCredentials(
                            clusterConfig.getUser(),
                            clusterConfig.getToken()),
                    rpcOptions);
        });
    }

    private BusConnector getBusConnector() {
        if (busConnector == null) {
            synchronized (busConnectorSyncRoot) {
                if (busConnector == null) {
                    logger.info("Creating a bus connector");
                    busConnector = new DefaultBusConnector(new NioEventLoopGroup(), true);
                }
            }
        }
        return busConnector;
    }

    public boolean getIsClosed() {
        return isClosed.get();
    }

    public void close() throws Exception {
        isClosed.set(true);
        for (Map.Entry<String, YtClient> entry : clientMap.entrySet()) {
            logger.info("Closing client for cluster '{}'", entry.getKey());
            entry.getValue().close();
        }
        if (busConnector != null) {
            synchronized (busConnectorSyncRoot) {
                if (busConnector != null) {
                    busConnector.close();
                    busConnector = null;
                }
            }
        }

    }
}
