package ru.yandex.travel.commons.yt;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import com.google.protobuf.Message;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;

import ru.yandex.travel.commons.health.HealthChecked;
import ru.yandex.travel.commons.messaging.AbstractMessageBusAndKeyValueStorage;
import ru.yandex.travel.commons.messaging.ClusterHealthStatus;
import ru.yandex.travel.commons.messaging.CompressionSettings;
import ru.yandex.travel.commons.messaging.Envelope;


public class MultiClusterYtAdapter extends AbstractMessageBusAndKeyValueStorage implements HealthChecked {
    private final CompressionSettings compressionSettings;
    private final Timer combinedSendStallTimer;
    private final Timer combinedSendActiveTimer;
    private final Timer combinedSendTotalTimer;
    private final Counter bytesSentCounter;
    private final Counter envelopeSentCounter;
    private final Counter sendSuccessCounter;
    private final Counter sendFailCounter;
    private final Timer combinedReadTimer;
    private final Counter combinedReadSuccessCounter;
    private final Counter combinedReadFailCounter;
    private final Counter combinedReadEmptyCounter;

    private final List<SingleClusterYtAdapter> clusterAdapters;
    private int minWrites;
    private int minReads;

    public MultiClusterYtAdapter(ClientReplicatedYtProperties ytConfig, String metricPrefix,
                                 List<SingleClusterYtAdapter> clusterAdapters) {
        minWrites = ytConfig.getMinWritesToContinue();
        minReads = ytConfig.getMinReadsToContinue();
        compressionSettings = new CompressionSettings(ytConfig.getMessageCodec(), ytConfig.getCompressionLevel());
        this.clusterAdapters = clusterAdapters;
        // Initialize metrics.
        combinedSendStallTimer = YtAdapterMetricsHelper.createTimer(String.format("yt.%s.combined.sendStallTime",
                metricPrefix));
        combinedSendActiveTimer = YtAdapterMetricsHelper.createTimer(String.format("yt.%s.combined.sendActiveTime",
                metricPrefix));
        combinedSendTotalTimer = YtAdapterMetricsHelper.createTimerWithExtendedBuckets(
                String.format("yt.%s.combined.sendTotalTime", metricPrefix));

        bytesSentCounter = Metrics.counter(String.format("yt.%s.combined.bytesSent", metricPrefix));
        envelopeSentCounter = Metrics.counter(String.format("yt.%s.combined.envelopesSent", metricPrefix));
        sendSuccessCounter = Metrics.counter(String.format("yt.%s.combined.sendSucceded", metricPrefix));
        sendFailCounter = Metrics.counter(String.format("yt.%s.combined.sendFailed", metricPrefix));

        combinedReadTimer = YtAdapterMetricsHelper.createTimer(String.format("yt.%s.combined.readTime", metricPrefix));
        combinedReadSuccessCounter = Metrics.counter(String.format("yt.%s.combined.readSucceeded", metricPrefix));
        combinedReadFailCounter = Metrics.counter(String.format("yt.%s.combined.readFailed", metricPrefix));
        combinedReadEmptyCounter = Metrics.counter(String.format("yt.%s.combined.readEmpty", metricPrefix));

    }

    @Override
    public CompletableFuture<Void> send(Envelope envelope) {
        if (minWrites == 0) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> result = new CompletableFuture<>();
        envelopeSentCounter.increment();
        bytesSentCounter.increment(envelope.getBytes().length);
        long writeStartTime = System.currentTimeMillis();
        combinedSendStallTimer.record(writeStartTime - envelope.getTimestamp(), TimeUnit.MILLISECONDS);
        AtomicInteger errorCount = new AtomicInteger();
        AtomicInteger successCount = new AtomicInteger();
        clusterAdapters.forEach(cluster -> cluster.send(envelope).whenComplete((v, t) -> {
            long writeEndTime = System.currentTimeMillis();
            combinedSendActiveTimer.record(writeEndTime - writeStartTime, TimeUnit.MILLISECONDS);
            combinedSendTotalTimer.record(writeEndTime - envelope.getTimestamp(), TimeUnit.MILLISECONDS);
            if (t == null) {
                if (successCount.incrementAndGet() >= minWrites) {
                    sendSuccessCounter.increment();
                    result.complete(null);
                }
            } else {
                if (errorCount.incrementAndGet() > clusterAdapters.size() - minWrites) {
                    sendFailCounter.increment();
                    result.completeExceptionally(t);
                }
            }
        }));
        return result;
    }

    @Override
    public <T extends Message> CompletableFuture<T> get(String key, Class<? extends T> messageClass) {
        if (minReads == 0) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<T> result = new CompletableFuture<>();
        List<SingleClusterYtAdapter> clusters = new ArrayList<>(clusterAdapters);
        long readStartTime = System.currentTimeMillis();
        AtomicInteger errorCount = new AtomicInteger();
        AtomicInteger successCount = new AtomicInteger();
        AtomicReference<T> resultRef = new AtomicReference<>(null);
        result.whenComplete((r, t) -> {
            long readEndTime = System.currentTimeMillis();
            combinedReadTimer.record(readEndTime - readStartTime, TimeUnit.MILLISECONDS);
            if (r == null) {
                combinedReadEmptyCounter.increment();
            }
            if (t != null) {
                combinedReadFailCounter.increment();
            } else {
                combinedReadSuccessCounter.increment();
            }
        });
        clusters.forEach(cluster -> cluster.get(key, messageClass).whenComplete((r, t) -> {
            if (t == null) {
                if (r != null) {
                    resultRef.set(r);
                }
                if (successCount.incrementAndGet() >= minReads) {
                    result.complete(resultRef.get());
                }
            } else {
                if (errorCount.incrementAndGet() > clusters.size() - minReads) {
                    result.completeExceptionally(t);
                }
            }
        }));
        return result;
    }

    @Override
    public CompressionSettings getCompressionSettings() {
        return compressionSettings;
    }

    @Override
    public ClusterHealthStatus isAlive() {
        int aliveCount = 0;
        ClusterHealthStatus combinedHealth = new ClusterHealthStatus();
        combinedHealth.setDetails(new HashMap<>());
        for (SingleClusterYtAdapter s : clusterAdapters) {
            ClusterHealthStatus clusterHealth = s.isAlive();
            for (Map.Entry<String, String> e : clusterHealth.getDetails().entrySet()) {
                combinedHealth.getDetails().put(e.getKey(), e.getValue());
            }
            if (clusterHealth.isUp()) {
                ++aliveCount;
            }
        }
        if (aliveCount >= Math.max(minReads, minWrites)) {
            combinedHealth.setUp(true);
        } else {
            combinedHealth.setUp(false);
        }
        return combinedHealth;
    }

    public void close() {
        clusterAdapters.forEach(SingleClusterYtAdapter::stopHealthCheckThread);
    }

    public void startHealthCheckThread() {
        clusterAdapters.forEach(SingleClusterYtAdapter::startHealthCheckThread);
    }

    @Override
    public boolean isHealthy() {
        return isAlive().isUp();
    }
}
