package ru.yandex.travel.spring.tx;

import java.lang.reflect.UndeclaredThrowableException;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.dao.support.PersistenceExceptionTranslator;
import org.springframework.lang.Nullable;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionOperations;

// A copy of spring transaction template with a forced rollback
@Slf4j
public class ForcedRollbackTxTemplate extends DefaultTransactionDefinition implements TransactionOperations {

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    /**
     * @param txManagerHelper       Transaction Manager helper that can be switched to paused mode
     * @param transactionDefinition the transaction definition to copy the
     *                              default settings from. Local properties can still be set to change values.
     */
    public ForcedRollbackTxTemplate(ForcedRollbackTxManagerWrapper txManagerHelper,
                                    TransactionDefinition transactionDefinition) {
        super(transactionDefinition);
        this.forcedRollbackTxManagerWrapper =
                Preconditions.checkNotNull(txManagerHelper, "Transaction manager helper must be provided");
    }

    @Override
    @Nullable
    public <T> T execute(TransactionCallback<T> action) throws TransactionException {
        ForcedRollbackTxManagerWrapper.TransactionStatusWrapper txStatusWrapper =
                forcedRollbackTxManagerWrapper.getTransaction(this);
        T result;
        try {
            result = action.doInTransaction(txStatusWrapper.getTransactionStatus());
            // see the notes to the flush call in TrainToGenericMigrationProcessor.migrate
            // for more information on wh we have to perform it two times before the final commit
            // todo(tlg-13): caused some unexpectedly heavy load from task processors,
            // we've decided to keep the hotfix only for workflow handlers
            //txStatusWrapper.getTransactionStatus().flush();
        } catch (RuntimeException | Error ex) {
            // Transactional code threw application exception -> rollback
            forcedRollbackTxManagerWrapper.rollbackOnException(txStatusWrapper, ex);
            throw translateIfPossibleAndRethrow(ex);
        } catch (Throwable ex) {
            // Transactional code threw unexpected exception -> rollback
            forcedRollbackTxManagerWrapper.rollbackOnException(txStatusWrapper, ex);
            throw new UndeclaredThrowableException(ex, "TransactionCallback threw undeclared checked exception");
        }
        forcedRollbackTxManagerWrapper.commitTransaction(txStatusWrapper);
        return result;
    }

    /**
     * We don't return Throwable (the common ancestor of Error and RuntimeException) from this method as the caller
     * won't be able to re-throw it in a simple way. The returned type is specified to let the caller syntactically
     * define what is going to happen during the call (no code after a call of this method call can be ever executed).
     */
    private RuntimeException translateIfPossibleAndRethrow(Throwable t) {
        if (!(t instanceof RuntimeException)) {
            // only Errors and RuntimeExceptions are passed to this method
            throw (Error) t;
        }
        if (t instanceof IllegalArgumentException | t instanceof IllegalStateException) {
            // no special meaning in these exceptions
            // but we still don't want to wrap them in InvalidDataAccessApiUsageException by the code below
            throw (RuntimeException) t;
        }
        // Spring translates exceptions when it can intercept them (repositories, entity and transaction managers),
        // but there are cases when Hibernate entity proxies throw specific exceptions when unable to load from the db
        // (e.g. constant locks during repeating db migration attempts) before any intercepting code (e.g. commit) is
        // called. We still need to get generalized Spring exceptions like ConcurrencyFailureException on the outside.
        PersistenceExceptionTranslator exceptionTranslator = forcedRollbackTxManagerWrapper.getExceptionTranslator();
        throw DataAccessUtils.translateIfNecessary((RuntimeException) t, exceptionTranslator);
    }

    public RuntimeException translateIfPossibleAndRethrow(RuntimeException e) {
        throw translateIfPossibleAndRethrow((Throwable) e);
    }
}
