package ru.yandex.travel.orders.client;

import java.net.InetAddress;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.grpc.MethodDescriptor;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;

import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.grpc.interceptors.UserCredentialsClientInterceptor;
import ru.yandex.travel.tvm.TvmWrapper;

@RequiredArgsConstructor
public abstract class GrpcServiceConfigurationBase {
    protected final GrpcChannelSupplierPropertiesBase properties;
    protected final Tracer tracer;

    public HAGrpcChannelFactory getGrpcChannelFactory(TvmWrapper tvm,
                                                      Function<Channel, CompletableFuture<ChannelState>> pingProducer,
                                                      Consumer<ManagedChannelBuilder<?>> patchChannelBuilder,
                                                      List<io.grpc.MethodDescriptor<?, ?>> methodsIgnoredInLog) {
        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder
                .withPingProducer(pingProducer)
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(new GrpcChannelSupplierFactory(properties).getChannelSupplier())
                .withChannelBuilder(label -> this.createChannel(label, getTvmHelper(tvm), patchChannelBuilder,
                        methodsIgnoredInLog))
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target,
                                         TvmHeaderClientHelper tvmHeaderClientHelper,
                                         Consumer<ManagedChannelBuilder<?>> patchChannelBuilder,
                                         List<io.grpc.MethodDescriptor<?, ?>> methodsIgnoredInLog) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());

        var ignoredMethods = methodsIgnoredInLog.stream().map(MethodDescriptor::getFullMethodName)
                .collect(Collectors.toUnmodifiableSet());
        var loggingClientInterceptor = new LoggingClientInterceptor(clientFqdn, target, ignoredMethods, tracer);
        var defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(properties.getTimeout());
        var tracerClientInterceptor = new TracerClientInterceptor(tracer);
        ClientInterceptor tmpClientInterceptor = null;
        if (tvmHeaderClientHelper != null) {
            tmpClientInterceptor = tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias());
        }
        UserCredentialsClientInterceptor userCredentialsClientInterceptor = null;
        if (properties.isForwardUserCredentials()) {
            userCredentialsClientInterceptor = new UserCredentialsClientInterceptor();
        }
        List<ClientInterceptor> interceptors = Stream.of(
                loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor, tmpClientInterceptor, userCredentialsClientInterceptor
        ).filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());


        var managedChannelBuilder = ManagedChannelBuilder.forTarget(target)
                .intercept(interceptors)
                .usePlaintext();

        if (patchChannelBuilder != null) {
            patchChannelBuilder.accept(managedChannelBuilder);
        }

        return new LabeledChannel(target, managedChannelBuilder.build());
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }
}
