package ru.yandex.travel.orders.client;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.client.yp.YpChannelSupplier;

@Slf4j
public class GrpcChannelSupplierFactory {
    private final String channelSupplierName;
    private final GrpcChannelSupplierProperties grpcChannelSupplierProperties;

    public GrpcChannelSupplierFactory(GrpcChannelSupplierProperties properties) {
        channelSupplierName = properties.getClass().getSimpleName().split("\\Q$\\E")[0];
        grpcChannelSupplierProperties = properties;
    }

    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierMode mode = grpcChannelSupplierProperties.getMode();

        Preconditions.checkState(mode != null, channelSupplierName + " has not channel supplier mode. `mode` is required");

        switch (mode) {
            case TARGETS:
                return getFixedListChannelSupplier();
            case YP:
                return getYpChannelSupplier();
            case EMPTY:
            default:
                return getEmptyChannelSupplier();
        }
    }

    private YpChannelSupplier getYpChannelSupplier() {
        Preconditions.checkState(
                grpcChannelSupplierProperties.getYp() != null,
                channelSupplierName + " in YP mode. Yp settings are required for grpc channel supplier."
        );

        log.info(channelSupplierName + " will use YP discovery grpc channel supplier");

        return new YpChannelSupplier(grpcChannelSupplierProperties.getYp());
    }

    private FixedListChannelSupplier getFixedListChannelSupplier() {
        Preconditions.checkState(
                grpcChannelSupplierProperties.getTargets() != null,
                channelSupplierName + " in TARGET mode. Target settings are required for grpc channel supplier."
        );

        log.info(channelSupplierName + " will use fixed list grpc channel supplier");

        return new FixedListChannelSupplier(grpcChannelSupplierProperties.getTargets());
    }

    private EmptyChannelSupplier getEmptyChannelSupplier() {
        log.info(channelSupplierName + " will use empty channel supplier");

        return new EmptyChannelSupplier();
    }
}
