package ru.yandex.travel.orders.client.yp;

import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class LocalYpCache {
    protected ConcurrentHashMap<String, List<Endpoint>> cachedMap;
    private final String filename;
    private final ObjectMapper objectMapper = new ObjectMapper();

    public LocalYpCache(String filename) {
        this.filename = filename;
    }


    public void put(String locationName, List<Endpoint> endpoints) {
        cachedMap.put(locationName, endpoints);
    }

    public List<Endpoint> get(String locationName) {
        return Collections.unmodifiableList(cachedMap.getOrDefault(locationName, Collections.emptyList()));
    }

    public synchronized void save() {
        try {
            String serialized = objectMapper.writerFor(cachedMap.getClass()).writeValueAsString(cachedMap);
            Files.writeString(Path.of(filename), serialized);
        } catch (Exception e) {
            log.error("Unable to save local cache", e);
        }
    }

    public synchronized void load() {
        var reader = objectMapper.readerFor(objectMapper.getTypeFactory().constructMapType(ConcurrentHashMap.class, objectMapper.getTypeFactory().constructType(String.class),
                objectMapper.getTypeFactory().constructCollectionLikeType(List.class, Endpoint.class)));
        Path path = Path.of(filename);
        try {
            String fileContent = Files.readString(path);
            cachedMap = reader.readValue(fileContent);
        } catch (NoSuchFileException e) {
            log.warn("Local cache file not found (path: {})", path);
            cachedMap = new ConcurrentHashMap<>();
        } catch (Exception e) {
            log.error("Unable to read local cache", e);
            cachedMap = new ConcurrentHashMap<>();
        }
    }
}
