package ru.yandex.travel.orders.client.yp;

import java.io.UncheckedIOException;
import java.text.MessageFormat;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.yp.YpRawClient;
import ru.yandex.yp.YpRawClientBuilder;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpSelectedObjects;

@Slf4j
public class YpChannelSupplier extends ChannelSupplier {
    private final Map<String, YpRawClient> ypClientsMap;
    private final String endpointSetId;
    private final LocalYpCache localYpCache;

    private final ScheduledExecutorService executorService;


    public YpChannelSupplier(YpDiscoveryProperties discoveryProperties) {
        this(discoveryProperties.getLocations().stream()
                        .collect(Collectors.toMap(l -> l, l -> new YpRawClientBuilder(YpLocation.valueOf(l).getInstance(), discoveryProperties::getToken).build())),
                new LocalYpCache(discoveryProperties.getLocalCachePath()), discoveryProperties.getEndpointSetId(),
                Executors.newSingleThreadScheduledExecutor(
                        new ThreadFactoryBuilder()
                                .setDaemon(true)
                                .setNameFormat("YpChannelDiscovery")
                                .build()));
        this.executorService.scheduleWithFixedDelay(this::refresh, discoveryProperties.getInitialRefreshDelay().toMillis(),
                discoveryProperties.getRefreshDelay().toMillis(), TimeUnit.MILLISECONDS);
    }

    YpChannelSupplier(Map<String, YpRawClient> ypClients, LocalYpCache localYpCache, String endpointSetId,
                      ScheduledExecutorService executorService) {
        this.ypClientsMap = ypClients;
        this.endpointSetId = endpointSetId;
        this.localYpCache = localYpCache;
        this.localYpCache.load();
        this.executorService = executorService;
    }

    public void close() throws InterruptedException {
        log.info("Stopping YP channel supplier");
        executorService.shutdown();
        executorService.awaitTermination(1, TimeUnit.SECONDS);
        ypClientsMap.values().forEach(client -> {
            if (!client.shutdown(1, TimeUnit.SECONDS)) {
                client.shutdownNow(100, TimeUnit.MILLISECONDS);
            }
        });
    }

    protected CompletableFuture<List<Endpoint>> discover(YpRawClient client) {
        YpSelectStatement statement = YpSelectStatement.protobufBuilder(YpObjectType.ENDPOINT)
                .addSelector("/meta")
                .addSelector("/spec")
                .setFilter(MessageFormat.format("([/meta/endpoint_set_id] = \"{0}\")", endpointSetId))
                .setOffset(0)
                .setLimit(100)
                .build();
        return client.objectService().selectObjects(statement, attributes -> {
            try {
                Autogen.TEndpointMeta meta = Autogen.TEndpointMeta.parseFrom(attributes.get(0).getProtobuf().get());
                DataModel.TEndpointSpec spec = DataModel.TEndpointSpec.parseFrom(attributes.get(1).getProtobuf().get());
                return new Endpoint(meta.getId(), spec.hasFqdn() ? spec.getFqdn() : null, spec.hasPort() ? spec.getPort() : null);
            } catch (InvalidProtocolBufferException e) {
                throw new UncheckedIOException(e);
            }
        }).thenApply(YpSelectedObjects::getResults);
    }

    public void refresh() {
        try {
            refreshImpl();
        } catch (Throwable ex) {
            log.error("EP refresh failed", ex);
        }
    }

    private void refreshImpl() {
        log.info("Refreshing YP endpoints");
        CompletableFuture.allOf(ypClientsMap.entrySet().stream().map(entry -> {
            String location = entry.getKey();
            YpRawClient client = entry.getValue();
            return discover(client)
                    .exceptionally(ex ->
                    {
                        log.warn("YP discovery failed for location {}, will use cached value", location, ex);
                        return localYpCache.get(location);
                    })
                    .whenComplete((res, t) -> {
                        if (t != null) {
                            log.error("Unexpected error on discovery", t);
                        } else {
                            var oldEps = new HashSet<>(localYpCache.get(location));
                            oldEps.removeAll(res);
                            oldEps.forEach(ep -> onChannelLost(ep.getTarget()));
                            res.forEach(ep -> this.onChannelDiscovered(ep.getTarget()));
                            localYpCache.put(location, res);
                        }
                    });
        }).toArray(CompletableFuture[]::new)).thenAccept(ignored -> localYpCache.save()).join();
    }
}
