package ru.yandex.travel.hibernate.types;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Properties;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.internal.util.ReflectHelper;
import org.hibernate.type.StringType;
import org.hibernate.usertype.DynamicParameterizedType;
import org.hibernate.usertype.EnhancedUserType;
import org.hibernate.usertype.UserType;

@Slf4j
@SuppressWarnings("unchecked")
public class CustomEnumType implements UserType, DynamicParameterizedType, EnhancedUserType {
    private static final String DEFAULT_PARSE_METHOD_NAME = "fromString";
    private static final String DEFAULT_VALUE_METHOD_NAME = "getValue";

    private static final String PARSE_METHOD_PARAMETER_NAME = "parse_method";
    private static final String VALUE_METHOD_PARAMETER_NAME = "value_method";

    private Class<? extends Enum> enumClass;
    public static final String ENUM = "enumClass";

    private BiMap<String, ? extends Enum> mapping;

    @SuppressWarnings("Duplicates")
    @Override
    public void setParameterValues(Properties parameters) {
        final DynamicParameterizedType.ParameterType reader = (DynamicParameterizedType.ParameterType) parameters.get(PARAMETER_TYPE);

        if (reader != null) {
            enumClass = reader.getReturnedClass().asSubclass(Enum.class);
        } else {
            final String enumClassName = (String) parameters.get(ENUM);
            try {
                enumClass = ReflectHelper.classForName(enumClassName, this.getClass()).asSubclass(Enum.class);
            } catch (ClassNotFoundException exception) {
                throw new HibernateException("Enum class not found: " + enumClassName, exception);
            }
        }

        ImmutableBiMap.Builder<String, Enum> mappingBuilder = ImmutableBiMap.builder();

        String valueMethodName = parameters.getProperty(VALUE_METHOD_PARAMETER_NAME, DEFAULT_VALUE_METHOD_NAME);
        Method valueMethod;
        try {
            valueMethod = enumClass.getMethod(valueMethodName);
        } catch (NoSuchMethodException e) {
            throw new HibernateException(String.format(
                    "Enum class '%s' does not implement '%s' method", enumClass.getName(), valueMethodName));
        }
        if (valueMethod.getReturnType() != String.class) {
            throw new HibernateException(String.format(
                    "Enum class '%s' method '%s' does not return a string", enumClass.getName(), valueMethodName));
        }

        for (Enum constant : enumClass.getEnumConstants()) {
            try {
                String value = (String) valueMethod.invoke(constant);
                mappingBuilder.put(value, constant);
            } catch (IllegalAccessException | InvocationTargetException e) {
                throw new HibernateException(e);
            }
        }

        mapping = mappingBuilder.build();
    }

    @Override
    public int[] sqlTypes() {
        return new int[]{Types.VARCHAR};
    }

    @Override
    public Class returnedClass() {
        return enumClass;
    }

    @Override
    public boolean equals(Object x, Object y) throws HibernateException {
        return x == y; // enums!
    }

    @Override
    public int hashCode(Object x) throws HibernateException {
        return x == null ? 0 : x.hashCode();
    }

    @Override
    public Object nullSafeGet(ResultSet rs, String[] names, SharedSessionContractImplementor session, Object owner) throws HibernateException, SQLException {
        String strValue = (String) StringType.INSTANCE.get(rs, names[0], session);
        if (strValue == null) {
            return null;
        }
        return mapping.get(strValue);
    }

    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index, SharedSessionContractImplementor session) throws HibernateException, SQLException {
        if (value == null) {
            StringType.INSTANCE.nullSafeSet(st, null, index, session);
        } else {
            StringType.INSTANCE.nullSafeSet(st, mapping.inverse().get(value), index, session);
        }
    }

    @Override
    public Object deepCopy(Object value) throws HibernateException {
        return value;
    }

    @Override
    public boolean isMutable() {
        return false;
    }

    @Override
    public Serializable disassemble(Object value) throws HibernateException {
        return (Serializable) value;
    }

    @Override
    public Object assemble(Serializable cached, Object owner) throws HibernateException {
        return cached;
    }

    @Override
    public Object replace(Object original, Object target, Object owner) throws HibernateException {
        return original;
    }

    @Override
    public String objectToSQLString(Object value) {
        return mapping.inverse().get(value);
    }

    @Override
    @SuppressWarnings("deprecation")
    public String toXMLString(Object value) {
        return mapping.inverse().get(value);
    }

    @Override
    @SuppressWarnings("deprecation")
    public Object fromXMLString(String xmlValue) {
        return mapping.get(xmlValue);
    }
}
