package ru.yandex.travel.hibernate.types;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ProtocolMessageEnum;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.internal.util.ReflectHelper;
import org.hibernate.type.IntegerType;
import org.hibernate.usertype.DynamicParameterizedType;
import org.hibernate.usertype.UserType;

public class ProtobufEnumType implements UserType, DynamicParameterizedType {
    public static final String ENUM = "enumClass";

    private static ConcurrentMap<Class<? extends ProtocolMessageEnum>, BiMap<Integer, ? extends ProtocolMessageEnum>>
            globalProtobufEnumMappings = new ConcurrentHashMap<>();


    public static Map<Class<? extends ProtocolMessageEnum>, BiMap<Integer, ? extends ProtocolMessageEnum>> getGlobalMapping() {
        return ImmutableMap.copyOf(globalProtobufEnumMappings);
    }

    private Class<? extends ProtocolMessageEnum> enumClass;
    private BiMap<Integer, ? extends ProtocolMessageEnum> mapping;

    @Override
    public int[] sqlTypes() {
        return new int[]{Types.INTEGER};
    }

    @Override
    public Class returnedClass() {
        return enumClass;
    }

    @Override
    public boolean equals(Object x, Object y) throws HibernateException {
        return x == y;
    }

    @Override
    public int hashCode(Object x) throws HibernateException {
        return x == null ? 0 : x.hashCode();
    }

    @Override
    public Object nullSafeGet(ResultSet rs, String[] names, SharedSessionContractImplementor session, Object owner) throws HibernateException, SQLException {
        Integer value = (Integer) IntegerType.INSTANCE.get(rs, names[0], session);
        if (value == null) {
            return null;
        }
        return mapping.get(value);
    }

    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index, SharedSessionContractImplementor session) throws HibernateException, SQLException {

        if (value == null) {
            IntegerType.INSTANCE.nullSafeSet(st, null, index, session);
        } else {
            var val = (ProtocolMessageEnum) value;
            IntegerType.INSTANCE.nullSafeSet(st, mapping.inverse().get(val), index, session);
        }
    }

    @Override
    public Object deepCopy(Object value) throws HibernateException {
        return value;
    }

    @Override
    public boolean isMutable() {
        return false;
    }

    @Override
    public Serializable disassemble(Object value) throws HibernateException {
        return (Serializable) value;
    }

    @Override
    public Object assemble(Serializable cached, Object owner) throws HibernateException {
        return cached;
    }

    @Override
    public Object replace(Object original, Object target, Object owner) throws HibernateException {
        return original;
    }


    @SuppressWarnings("Duplicates")
    @Override
    public void setParameterValues(Properties parameters) {
        final DynamicParameterizedType.ParameterType reader =
                (DynamicParameterizedType.ParameterType) parameters.get(PARAMETER_TYPE);

        if (reader != null) {
            enumClass = reader.getReturnedClass().asSubclass(Enum.class);
        } else {
            final String enumClassName = (String) parameters.get(ENUM);
            try {
                enumClass = ReflectHelper.classForName(enumClassName, this.getClass()).asSubclass(Enum.class);
            } catch (ClassNotFoundException exception) {
                throw new HibernateException("Enum class not found: " + enumClassName, exception);
            }
        }

        ImmutableBiMap.Builder<Integer, ProtocolMessageEnum> mappingBuilder = ImmutableBiMap.builder();

        String valueMethodName = "getNumber";
        Method valueMethod;
        try {
            valueMethod = enumClass.getMethod(valueMethodName);
        } catch (NoSuchMethodException e) {
            throw new HibernateException(String.format(
                    "Enum class '%s' does not implement '%s' method", enumClass.getName(), valueMethodName));
        }
        if (valueMethod.getReturnType() != int.class) {
            throw new HibernateException(String.format(
                    "Enum class '%s' method '%s' does not return an integer", enumClass.getName(), valueMethodName));
        }

        for (ProtocolMessageEnum constant : enumClass.getEnumConstants()) {
            if ("UNRECOGNIZED".equals(constant.toString())) {
                continue;
            }
            try {
                Integer value = (Integer) valueMethod.invoke(constant);
                mappingBuilder.put(value, constant);
            } catch (IllegalAccessException | InvocationTargetException e) {
                throw new HibernateException(e);
            }
        }

        mapping = mappingBuilder.build();
        globalProtobufEnumMappings.putIfAbsent(enumClass, mapping);
    }
}
