package ru.yandex.travel.hibernate.types;

import com.google.common.base.Verify;
import com.google.protobuf.Message;
import lombok.Getter;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.type.BinaryType;
import org.hibernate.type.ClassType;
import org.hibernate.type.Type;
import org.hibernate.usertype.CompositeUserType;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class ProtobufMessageType implements CompositeUserType {
    @Override
    public String[] getPropertyNames() {
        return new String[]{"class_name", "name"};
    }

    @Override
    public Type[] getPropertyTypes() {
        return new Type[]{
                ClassType.INSTANCE, BinaryType.INSTANCE
        };
    }

    @Override
    public Object getPropertyValue(Object component, int property) throws HibernateException {
        if (component == null) {
            return null;
        }
        switch (property) {
            case 0:
                return component.getClass();
            case 1:
                return ((Message) component).toByteArray();
            default:
                throw new HibernateException("Invalid property index [" + property + "]");
        }
    }

    @Override
    public void setPropertyValue(Object component, int property, Object value) throws HibernateException {
        throw new HibernateException("Protobuf message is immutable, so properties can't be set. Rewrite the value");
    }

    @Override
    public Class returnedClass() {
        return Object.class; //maybe we can assume Message interface descendant
    }

    @Override
    public boolean equals(Object x, Object y) throws HibernateException {
        if (x == null && y == null) {
            return true;
        }
        if (x != null && y != null) {
            return x.equals(y);
        }
        return false;
    }

    @Override
    public int hashCode(Object x) throws HibernateException {
        return x == null ? 0 : x.hashCode();
    }

    @Override
    @SuppressWarnings("unchecked")
    public Object nullSafeGet(ResultSet rs, String[] names, SharedSessionContractImplementor session, Object owner) throws HibernateException, SQLException {
        Verify.verify(names.length == 2);
        Class clazz = (Class) ClassType.INSTANCE.get(rs, names[0], session);
        byte[] data = (byte[]) BinaryType.INSTANCE.get(rs, names[1], session);
        if (clazz == null && data == null) {
            return null;
        }
        if (clazz == null) {
            throw new HibernateException("Wrong situation, there's non-null data and no class given");
        }
        if (!Message.class.isAssignableFrom(clazz)) {
            throw new HibernateException(String.format("Can't assign Message class from %s", clazz));
        }
        try {
            return ProtobufUtils.parseFrom(clazz, data);
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            throw new HibernateException(e);
        }
    }

    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index, SharedSessionContractImplementor session) throws HibernateException, SQLException {
        if (value == null) {
            ClassType.INSTANCE.set(st, null, index, session);
            BinaryType.INSTANCE.set(st, null, index + 1, session);
        } else {
            ClassType.INSTANCE.set(st, value.getClass(), index, session);
            BinaryType.INSTANCE.set(st, ((Message) value).toByteArray(), index + 1, session);
        }
    }

    @Override
    public Object deepCopy(Object value) throws HibernateException {
        if (value == null) {
            return null;    
        }
        return ((Message) value).toBuilder().build();
    }

    @Override
    public boolean isMutable() {
        return false;
    }

    @Override
    public Serializable disassemble(Object value, SharedSessionContractImplementor session) throws HibernateException {
        return new SerializableWrapper(
                value.getClass(),
                ((Message) value).toByteArray()
        );
    }

    @Override
    @SuppressWarnings("unchecked")
    public Object assemble(Serializable cached, SharedSessionContractImplementor session, Object owner) throws HibernateException {
        SerializableWrapper wrappedValue = (SerializableWrapper) cached;
        try {
            return ProtobufUtils.parseFrom(wrappedValue.getClazz(), wrappedValue.getData());
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            throw new HibernateException(e);
        }
    }

    @Override
    public Object replace(Object original, Object target, SharedSessionContractImplementor session, Object owner) throws HibernateException {
        return original;
    }

    private static final class SerializableWrapper implements Serializable {

        private static final long serialVersionUID = 1L;

        @Getter
        private Class clazz;
        @Getter
        private byte[] data;

        SerializableWrapper(Class clazz, byte[] data) {
            this.clazz = clazz;
            this.data = data;
        }
    }
}
