package ru.yandex.travel.commons.logging.ydb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.TableRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;

import ru.yandex.travel.logging.ydb.TOrderLogRecord;

public class DefaultYdbLogService implements YdbLogService{
    private final YdbLogTableClient logTableClient;

    public DefaultYdbLogService(YdbLogProperties properties) {
        Collection<AutoCloseable> closableResources = new ArrayList<>();
        try {
            AuthProvider authProvider = createTvmAuthProvider(properties);
            closableResources.add(authProvider);
            RpcTransport transport = createTransport(authProvider, properties);
            closableResources.add(transport);
            TableRpc tableRpc = GrpcTableRpc.ownTransport(transport);
            closableResources.add(tableRpc);
            TableClient tableClient = TableClient.newClient(tableRpc).build();
            closableResources.add(tableClient);
            this.logTableClient = new YdbLogTableClient(
                    authProvider,
                    tableClient,
                    properties.getTablePath(),
                    properties.getClientTimeout(),
                    properties.getMaxAttempts(),
                    properties.getBackoffSlot(),
                    properties.getBackoffCeiling());
        } catch (Exception e) {
            for (AutoCloseable resource : closableResources) {
                try {
                    resource.close();
                } catch (Exception ne) {
                    e.addSuppressed(ne);
                }
            }
            throw e;
        }
    }

    @Override
    public List<TOrderLogRecord> getLogsForIds(Set<UUID> uuidSet, String level, String logger,
                                               Integer offset, Integer limit, String searchText) {
        if (uuidSet == null || uuidSet.isEmpty()) {
            return List.of();
        }
        List<String> idList = uuidSet.stream()
                .map(UUID::toString)
                .collect(Collectors.toUnmodifiableList());
        return logTableClient.getLogRecords(idList, level, logger, offset, limit, searchText).join();
    }

    @Override
    public Long countLogsForIds(Set<UUID> uuidSet, String level, String logger, String searchText) {
        if (uuidSet == null || uuidSet.isEmpty()) {
            return 0L;
        }
        Set<String> idSet = uuidSet.stream()
                .map(UUID::toString)
                .collect(Collectors.toSet());
        return logTableClient.countLogRecords(idSet, level, logger, searchText).join();
    }

    private static AuthProvider createTvmAuthProvider(YdbLogProperties properties) {
        return YdbLogTvmAuthContext.tvmAuthProvider(properties.getTvmClientId(), properties.getTvmSecret());
    }

    private static RpcTransport createTransport(AuthProvider authProvider, YdbLogProperties properties) {
        return GrpcTransport.forEndpoint(properties.getDbEndpoint(), properties.getDbPath())
                .withAuthProvider(authProvider)
                .withReadTimeout(properties.getClientTimeout())
                .build();
    }
}
