package ru.yandex.travel.commons.logging.ydb;

import java.io.Serializable;
import java.time.Duration;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Plugin(name = "YdbLogAppender", category = "Core", elementType = "appender", printObject = true)
public class YdbLogAppender extends AbstractAppender {
    private static final String DEFAULT_LOG_TABLE = "order_logs";
    private static final int DEFAULT_QUEUE_SIZE = 16384;
    private static final int DEFAULT_BATCH_SIZE = 100;
    private static final int DEFAULT_MAX_ATTEMPTS = 10;
    private static final Duration DEFAULT_CLIENT_TIMEOUT = Duration.ofSeconds(10);
    private static final Duration DEFAULT_BACKOFF_SLOT = Duration.ofMillis(100);
    private static final int DEFAULT_BACKOFF_CEILING = 7;
    private static final Duration DEFAULT_SHUTDOWN_TIMEOUT = Duration.ofSeconds(60);

    private final YdbLogManager manager;

    private static final Logger log = LoggerFactory.getLogger("ru.yandex.travel.commons.logging.ydb");

    @SuppressWarnings("deprecation")
    public YdbLogAppender(String name, Filter filter, Layout<? extends Serializable> layout,
                          boolean ignoreExceptions, YdbLogManager manager) {
        super(name, filter, layout, ignoreExceptions);
        Preconditions.checkNotNull(manager, "manager can't be null");
        this.manager = manager;
    }

    @PluginFactory
    public static AbstractAppender createAppender(
            @PluginAttribute("name") String name,
            @PluginAttribute("ignoreExceptions") boolean ignoreExceptions,
            @PluginAttribute("enabled") boolean enabled,
            @PluginAttribute("dbEndpoint") String dbEndpoint,
            @PluginAttribute("dbPath") String dbPath,
            @PluginAttribute("tablePath") String tablePath,
            @PluginAttribute("tvmClientId") int tvmClientId,
            @PluginAttribute("tvmSecret") String tvmSecret,
            @PluginAttribute("queueSize") int queueSize,
            @PluginAttribute("batchSize") int batchSize,
            @PluginAttribute("maxAttempts") int maxAttempts,
            @PluginAttribute("clientTimeout") String clientTimeout,
            @PluginAttribute("backoffSlot") String backoffSlot,
            @PluginAttribute("backoffCeiling") int backoffCeiling,
            @PluginAttribute("shutdownTimeout") String shutdownTimeout,
            @PluginElement("Filters") Filter filter
    ) {
        if (!enabled) {
            log.warn("YDB logging is disabled");
            return new AbstractAppender(name, filter, null) {
                @Override
                public void append(LogEvent event) {
                    // not doing anything
                }
            };
        }

        log.info("Instantiating YdbLogAppender; base configuration: name={}, dbEndpoint={}, dbPath={}, tablePath={}, " +
                        "tvmClientId={}, queueSize={}, batchSize={}, maxAttempts={}, clientTimeout={}, shutdownTimeout={}",
                name, dbEndpoint, dbPath, tablePath, tvmClientId, queueSize, batchSize, maxAttempts, clientTimeout, shutdownTimeout);
        if (filter == null) {
            log.warn("No event Filter. At least YdbLogFilter should be configured");
        }
        Preconditions.checkArgument(!Strings.isNullOrEmpty(name), "No name provided for YdbAppender");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(dbEndpoint), "dbEndpoint is not specified");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(dbPath), "dbPath is not specified");
        Preconditions.checkArgument(tvmClientId > 0, "tvmClientId is not specified: %s", tvmClientId);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(tvmSecret), "tvmSecret is not specified");
        YdbLogProperties properties = YdbLogProperties.builder()
                .dbEndpoint(dbEndpoint)
                .dbPath(dbPath)
                .tablePath(!Strings.isNullOrEmpty(tablePath) ? tablePath : DEFAULT_LOG_TABLE)
                .tvmClientId(tvmClientId)
                .tvmSecret(tvmSecret)
                .queueSize(queueSize > 0 ? queueSize : DEFAULT_QUEUE_SIZE)
                .batchSize(batchSize > 0 ? batchSize : DEFAULT_BATCH_SIZE)
                .maxAttempts(maxAttempts > 0 ? maxAttempts : DEFAULT_MAX_ATTEMPTS)
                .clientTimeout(!Strings.isNullOrEmpty(clientTimeout) ?
                        Duration.parse(clientTimeout) :
                        DEFAULT_CLIENT_TIMEOUT)
                .backoffSlot(!Strings.isNullOrEmpty(backoffSlot) ?
                        Duration.parse(backoffSlot) :
                        DEFAULT_BACKOFF_SLOT)
                .backoffCeiling(backoffCeiling > 0 ? backoffCeiling : DEFAULT_BACKOFF_CEILING)
                .shutdownTimeout(!Strings.isNullOrEmpty(shutdownTimeout) ?
                        Duration.parse(shutdownTimeout) :
                        DEFAULT_SHUTDOWN_TIMEOUT)
                .build();
        log.info("Effective YDB log configuration: {}", properties);
        YdbLogManager manager = YdbLogManager.getManager(name, new YdbLogManagerFactory(), properties);
        return new YdbLogAppender(name, filter, null, ignoreExceptions, manager);
    }

    @Override
    public void start() {
        try {
            super.start();
            manager.start();
        } catch (Exception e) {
            try {
                // there are pre-initialized resources like TvmClient that should be freed in case of errors here
                manager.close();
            } catch (Exception ne) {
                e.addSuppressed(ne);
            }
            throw e;
        }
    }

    @Override
    public void append(LogEvent event) {
        manager.write(event);
    }

    @Override
    public boolean stop(long timeout, TimeUnit timeUnit) {
        boolean success = super.stop(timeout, timeUnit);
        success &= manager.stop(timeout, timeUnit);
        return success;
    }
}
