package ru.yandex.travel.commons.logging.ydb;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.codec.binary.Hex;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.pattern.ExtendedThrowablePatternConverter;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;

import ru.yandex.travel.commons.logging.masking.LogMaskingConverter;
import ru.yandex.travel.logging.ydb.TOrderLogRecord;

import static ru.yandex.travel.commons.logging.CommonMdcParams.MDC_ENTITY_ID;

public class YdbLogRecordFactory {
    private static final String MD5_DIGEST_NAME = "MD5";

    private static final String HOST_NAME_KEY = "HostName";

    private static final ObjectMapper jsonMapper = new ObjectMapper();
    private static final LogEventPatternConverter errorFormatter =
            ExtendedThrowablePatternConverter.newInstance(null, null);

    private final String hostName;
    private final LogMaskingConverter logMaskingConverter;

    public YdbLogRecordFactory(String hostName) {
        this.hostName = hostName;
        logMaskingConverter = LogMaskingConverter.create();
    }

    public TOrderLogRecord createFromLogEvent(LogEvent event) {
        StringBuilder messageBuilder = new StringBuilder();
        logMaskingConverter.format(event, messageBuilder);

        Set<Throwable> errors = new HashSet<>();
        errors.add(event.getThrown());
        errors.add(event.getMessage().getThrowable());
        for (Throwable error : errors) {
            if (error != null) {
                StringBuilder st = new StringBuilder("\n");
                errorFormatter.format(event, st);
                messageBuilder.append(st);
            }
        }

        String entityId = event.getContextData().getValue(MDC_ENTITY_ID);
        Map<String, String> contextCopyMap = new HashMap<>(event.getContextData().toMap());
        contextCopyMap.putIfAbsent(HOST_NAME_KEY, hostName);
        String context = toJson(contextCopyMap);
        String messageId = messageId(event);
        return TOrderLogRecord.newBuilder()
                .setOwnerId(entityId)
                .setTimestamp(event.getInstant().getEpochMillisecond())
                .setMessageId(messageId)
                .setLogger(event.getLoggerName())
                .setLevel(event.getLevel().name())
                .setMessage(messageBuilder.toString())
                .setContext(context)
                .build();
    }

    private String toJson(Object data) {
        try {
            return jsonMapper.writeValueAsString(data);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    String messageId(LogEvent event) {
        try {
            // using non-pk event fields to compute a semi-unique message hash
            MessageDigest md = MessageDigest.getInstance(MD5_DIGEST_NAME);
            md.update(toStringBytes(event.getLoggerFqcn()));
            md.update(toStringBytes(event.getMessage().getFormattedMessage().hashCode()));
            md.update(toStringBytes(event.getLevel()));
            md.update(toStringBytes(event.getMarker()));
            md.update(toStringBytes(event.getThreadName()));
            md.update(toStringBytes(hostName));
            if (event.getThrown() != null) {
                md.update(toStringBytes(event.getThrown().getMessage()));
            }
            if (event.getMessage().getThrowable() != null) {
                md.update(toStringBytes(event.getMessage().getThrowable().getMessage()));
            }
            return Hex.encodeHexString(md.digest());
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private static byte[] toStringBytes(Object value) {
        String str = value != null ? value.toString() : "null";
        return str.getBytes();
    }
}
