package ru.yandex.travel.workflow.ha;

import java.time.Duration;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

@Slf4j
public class HAManager implements InitializingBean, DisposableBean {

    private final MasterLockManager masterLockManager;

    private final AtomicReference<HAManagerState> state; // using AtomicReference as we access it from the gauge
    // supplier (can use volatile instead)

    private final AtomicReference<LockState> lockState;

    private final AtomicLong lastAcquisitionAttempt;

    private final ScheduledExecutorService scheduledExecutorService;

    private final ExecutorService stoppingExecutorService;

    private boolean finishing;

    private final Duration promoteDuration;

    private final Duration stopDuration;

    private final Duration retryAcquireLockInterval;

    private ScheduledFuture scheduledJob;

    private MasterStatusAwareResource masterStatusAwareResource;

    public HAManager(
            MasterLockManager masterLockManager,
            MasterStatusAwareResource masterStatusAwareResource,
            Duration promoteDuration, Duration stopDuration, Duration retryAcquireLockInterval
    ) {
        this.masterLockManager = masterLockManager;
        this.state = new AtomicReference<>(HAManagerState.STANDBY);
        this.lockState = new AtomicReference<>(LockState.lost());
        this.lastAcquisitionAttempt = new AtomicLong(System.currentTimeMillis());
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("ha-scheduler")
                        .setDaemon(true)
                        .build()
        );
        this.stoppingExecutorService = Executors.newSingleThreadExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("ha-stopper")
                        .setDaemon(true)
                        .build()
        );
        this.promoteDuration = promoteDuration;
        this.stopDuration = stopDuration;
        this.finishing = false;
        this.masterStatusAwareResource = masterStatusAwareResource;
        this.retryAcquireLockInterval = retryAcquireLockInterval;
        for (HAManagerState registeredState : HAManagerState.values()) {
            Gauge.builder("orchestrator.ha.manager.node", () -> state.get() == registeredState ? 1 : 0)
                    .tag("state", registeredState.toString()).register(Metrics.globalRegistry);
        }
        Gauge.builder("orchestrator.ha.manager.lockLostMs", this::timeSinceLockLost).register(Metrics.globalRegistry);
        Gauge.builder("orchestrator.ha.manager.lockAcquiredMs", this::timeSinceLockAcquired).register(Metrics.globalRegistry);
        Gauge.builder("orchestrator.ha.manager.lockIsAcquired", () -> lockState.get().isAcquired() ? 1 : 0)
                .register(Metrics.globalRegistry);
        Gauge.builder("orchestrator.ha.manager.timeSinceLastAcquisitionAttempt", this::timeSinceLastAcquisitionAttempt)
                .register(Metrics.globalRegistry);
    }

    @Override
    public void destroy() throws Exception {
        stop();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        tryAcquireLock();
    }

    private long timeSinceLockLost() {
        return lockState.get().timeSinceLost();
    }

    private long timeSinceLockAcquired() {
        return lockState.get().timeSinceAcquired();
    }

    private long timeSinceLastAcquisitionAttempt() {
        HAManagerState currentState = state.get();
        if (currentState == HAManagerState.STANDBY) {
            return System.currentTimeMillis() - lastAcquisitionAttempt.get();
        } else {
            return 0;
        }
    }

    private synchronized void stop() {
        finishing = true;
        HAManagerState currentState = state.get();
        Preconditions.checkState(
                currentState == HAManagerState.PROMOTING || currentState == HAManagerState.MASTER || currentState == HAManagerState.STANDBY
        );
        if (currentState == HAManagerState.PROMOTING || currentState == HAManagerState.MASTER) {
            onLockLost();
        } else {
            proceedToStandbyOrStopped();
        }
    }

    public HAManagerState getCurrentState() {
        return state.get();
    }

    private synchronized void promoteToMaster() {
        log.info("Master lock acquired: claiming to be the master");
        state.set(HAManagerState.MASTER);
        scheduledJob = null;
        masterStatusAwareResource.promotedToMaster();
    }

    private synchronized void onLockLost() {
        HAManagerState currentState = state.get();
        lockState.set(LockState.lost());
        Preconditions.checkState(currentState == HAManagerState.PROMOTING || currentState == HAManagerState.MASTER || currentState == HAManagerState.STOPPED);
        if (currentState == HAManagerState.PROMOTING) {
            log.info("Master lock lost: cancelling promotion");
            scheduledJob.cancel(true);
            proceedToStandbyOrStopped();
        } else if (currentState == HAManagerState.STOPPED) {
            log.info("Master lock lost: service was stopped already");
        } else {
            Preconditions.checkState(scheduledJob == null);
            log.info("Master lock lost: demoting service");
            state.set(HAManagerState.STOPPING);

            scheduledJob = scheduledExecutorService.schedule(this::forceStop, stopDuration.toNanos(),
                    TimeUnit.NANOSECONDS);
            stoppingExecutorService.submit(() -> {
                long stopStartTime = System.nanoTime();
                masterStatusAwareResource.prepareToStandby();
                long timePassed = System.nanoTime() - stopStartTime;
                lockLostNormalStop(timePassed);
            });
        }
    }

    private synchronized void forceStop() {
        log.info("Master lock lost: couldn't stop services in {}ms forcing stop", stopDuration.toMillis());
        state.set(HAManagerState.FORCED_STOP);
        scheduledJob = null;
        masterStatusAwareResource.forceStandby();
    }

    private synchronized void lockLostNormalStop(long shutdownTime) {
        HAManagerState currentState = state.get();
        Preconditions.checkState(currentState == HAManagerState.STOPPING || currentState == HAManagerState.FORCED_STOP);
        log.info("Master lock lost: stopped service in {}ms", Duration.ofNanos(shutdownTime).toMillis());
        if (state.get() == HAManagerState.STOPPING) {
            scheduledJob.cancel(true);
        }
        proceedToStandbyOrStopped();
    }

    private synchronized void proceedToStandbyOrStopped() {
        if (finishing) {
            state.set(HAManagerState.STOPPED);
            try {
                log.info("Stopping master status aware resources");
                masterStatusAwareResource.stopAll();
                log.info("Stopped master status aware resources");
            } catch (Exception e) {
                log.error("Exception stopping all master aware resources", e);
            }
            MoreExecutors.shutdownAndAwaitTermination(scheduledExecutorService, 2, TimeUnit.SECONDS);
            MoreExecutors.shutdownAndAwaitTermination(stoppingExecutorService, 2, TimeUnit.SECONDS);
        } else {
            state.set(HAManagerState.STANDBY);
            scheduledJob = scheduledExecutorService.schedule(this::tryAcquireLock, retryAcquireLockInterval.toNanos(),
                    TimeUnit.NANOSECONDS);
        }
    }

    private synchronized void tryAcquireLock() {
        HAManagerState currentState = state.get();
        log.debug("Trying to acquire master lock. Current state: {}", currentState);
        Preconditions.checkState(currentState == HAManagerState.STANDBY);
        boolean lockAcquired = false;
        try {
            lockAcquired = masterLockManager.acquireLock(this::onLockLost);
        } catch (Exception e) {
            log.error("Severe error acquiring lock", e);
        }
        lastAcquisitionAttempt.set(System.currentTimeMillis());
        if (lockAcquired) {
            state.set(HAManagerState.PROMOTING);
            lockState.set(LockState.acquired());
            scheduledJob = scheduledExecutorService.schedule(this::promoteToMaster, promoteDuration.toNanos(),
                    TimeUnit.NANOSECONDS);
        } else {
            scheduledJob = scheduledExecutorService.schedule(this::tryAcquireLock, retryAcquireLockInterval.toNanos()
                    , TimeUnit.NANOSECONDS);
        }
    }

    @Getter
    @RequiredArgsConstructor
    enum HAManagerState {
        MASTER, PROMOTING, STANDBY, STOPPING, FORCED_STOP, STOPPED
    }
}
