package ru.yandex.travel.clients.promo_service_booking_flow;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;

import com.google.common.base.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.KVExperiments;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TExperimentInfo;
import ru.yandex.travel.hotels.proto.THotelId;
import ru.yandex.travel.hotels.proto.TOfferInfo;
import ru.yandex.travel.hotels.proto.TUserInfo;
import ru.yandex.travel.hotels.proto.TWhiteLabelInfo;
import ru.yandex.travel.order_type.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.proto.BaseTypes;
import ru.yandex.travel.white_label.proto.EWhiteLabelPartnerId;

import static ru.yandex.travel.commons.proto.ProtoUtils.toTPrice;

public class PromoServiceBookingFlowUtils {
    private static final Map<EDisplayOrderType, EOrderType> orderTypeMap = Map.of(
            EDisplayOrderType.DT_AVIA, EOrderType.OT_AVIA,
            EDisplayOrderType.DT_BUS, EOrderType.OT_BUS,
            EDisplayOrderType.DT_HOTEL, EOrderType.OT_HOTEL,
            EDisplayOrderType.DT_SUBURBAN, EOrderType.OT_SUBURBAN,
            EDisplayOrderType.DT_TRAIN, EOrderType.OT_TRAIN
    );

    @NotNull
    public static TUserInfo buildUserInfo(String passportId, boolean userIsPlus) {
        return buildUserInfo(passportId, userIsPlus, Collections.emptyList(), false);
    }

    @NotNull
    public static TUserInfo buildUserInfo(
            String passportId,
            boolean userIsPlus,
            List<EOrderType> existingOrderTypes,
            boolean useExistingOrderTypes
    ) {
        TUserInfo.Builder userInfoBuilder = TUserInfo.newBuilder()
                .setIsPlus(userIsPlus)
                .setIsLoggedIn(!Strings.isNullOrEmpty(passportId))
                .addAllExistingOrderTypes(existingOrderTypes)
                .setUseExistingOrderTypes(useExistingOrderTypes);
        if (passportId != null) {
            userInfoBuilder.setPassportId(passportId);
        }
        return userInfoBuilder.build();
    }

    @NotNull
    public static TOfferInfo buildOfferInfo(
            EPartnerId partnerId,
            String originalId,
            String checkinDate,
            String checkoutDate,
            @Nonnull Money priceFromPartnerOffer,
            @Nullable Money priceBeforePromocodes,
            @Nullable Money priceAfterPromocodes
    ) {
        TOfferInfo.Builder builder = TOfferInfo.newBuilder()
                .setHotelId(THotelId.newBuilder()
                        .setPartnerId(partnerId)
                        .setOriginalId(originalId)
                        .build())
                .setCheckInDate(checkinDate)
                .setCheckOutDate(checkoutDate)
                .setPriceFromPartnerOffer(toTPrice(priceFromPartnerOffer, 0));
        if (priceBeforePromocodes != null) {
            builder.setPriceBeforePromocodes(toTPrice(priceBeforePromocodes, 0));
        }
        if (priceAfterPromocodes != null) {
            builder.setPriceAfterPromocodes(toTPrice(priceAfterPromocodes, 0));
        }
        return builder.build();
    }

    @NotNull
    public static TExperimentInfo buildExperimentInfo(ExperimentDataProvider experimentDataProvider, CommonHttpHeaders headers) {
        OrderExperiments orderExperiments = experimentDataProvider.getInstance(OrderExperiments.class, headers);
        return TExperimentInfo.newBuilder()
                .setStrikeThroughPrices(orderExperiments.isStrikeThroughPrices())
                .addAllKVExperiments(KVExperiments.toProto(headers.getExperiments()))
                .build();
    }

    @NotNull
    public static TExperimentInfo buildExperimentInfo(List<BaseTypes.TStringPair> kVExperiments) {
        return TExperimentInfo.newBuilder()
                .addAllKVExperiments(kVExperiments)
                .build();
    }

    @NotNull
    public static TWhiteLabelInfo buildWhiteLabelInfo(CommonHttpHeaders headers) {
        return TWhiteLabelInfo.newBuilder()
                .setPartnerId(headers.getWhiteLabelPartnerId())
                .build();
    }

    @NotNull
    public static TWhiteLabelInfo buildWhiteLabelInfo(EWhiteLabelPartnerId whiteLabelPartnerId) {
        return TWhiteLabelInfo.newBuilder()
                .setPartnerId(whiteLabelPartnerId)
                .build();
    }

    public static EOrderType mapDisplayOrderTypeToOrderType(EDisplayOrderType displayOrderType) {
        return orderTypeMap.get(displayOrderType);
    }

    public static List<EOrderType> mapDisplayOrderTypesToOrderTypes(List<EDisplayOrderType> displayOrderTypes) {
        return displayOrderTypes.stream()
                .map(orderTypeMap::get)
                .collect(Collectors.toList());
    }
}
