package ru.yandex.travel.hotels.services.promoservice;

import java.net.InetAddress;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.validation.constraints.NotNull;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.hotels.proto.PromoServiceV1Grpc;
import ru.yandex.travel.hotels.proto.TPingRpcReq;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@ConditionalOnBean(Tracer.class)
@EnableConfigurationProperties(PromoServiceConfigurationProperties.class)
@RequiredArgsConstructor
public class PromoServiceConfiguration {
    private final PromoServiceConfigurationProperties properties;
    private final Tracer tracer;

    @Bean
    public PromoServiceClient promoServiceClient(PromoServiceClientFactory clientFactory) {
        return new PromoServiceClientImpl(clientFactory, Clock.systemUTC());
    }

    @Bean
    public ChannelSupplier promoServiceChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean
    public HAGrpcChannelFactory promoServiceGrpcChannelFactory(@Autowired(required = false) TvmWrapper tvm) {
        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder.withPingProducer(promoServicePingProducer())
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(promoServiceChannelSupplier())
                .withChannelBuilder(label -> this.createChannel(label, getTvmHelper(tvm)))
                .build();
    }

    @NotNull
    private Function<Channel, CompletableFuture<ChannelState>> promoServicePingProducer() {
        return channel -> FutureUtils.buildCompletableFuture(PromoServiceV1Grpc
                .newFutureStub(channel)
                .ping(TPingRpcReq.newBuilder().build())
        ).thenApply(rsp -> rsp.getIsReady() ? ChannelState.READY_MASTER : ChannelState.NOT_READY);
    }

    @Bean
    public PromoServiceClientFactory promoServiceClientFactory(
            @Qualifier("promoServiceGrpcChannelFactory") HAGrpcChannelFactory channelFactory) {
        return () -> PromoServiceV1Grpc.newFutureStub(channelFactory.getRoundRobinChannel());
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(PromoServiceV1Grpc.getPingMethod().getFullMethodName()), tracer
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        TracerClientInterceptor tracerClientInterceptor = new TracerClientInterceptor(
                tracer
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor));
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }
}
