package ru.yandex.travel.integration.spark;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.net.HttpCookie;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPConstants;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;
import javax.xml.transform.stream.StreamSource;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.w3c.dom.Document;

import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.integration.spark.model.Authmethod;
import ru.yandex.travel.integration.spark.model.AuthmethodResponse;
import ru.yandex.travel.integration.spark.model.GetCompanyShortReport;
import ru.yandex.travel.integration.spark.model.GetCompanyShortReportResponse;
import ru.yandex.travel.integration.spark.model.GetEntrepreneurShortReport;
import ru.yandex.travel.integration.spark.model.GetEntrepreneurShortReportResponse;
import ru.yandex.travel.integration.spark.responses.CompanyShortReportResponse;
import ru.yandex.travel.integration.spark.responses.EntrepreneurShortReportResponse;

@Slf4j
public class SparkClient extends HttpApiClientBase {

    private static final String SESSION_HEADER_NAME = "ASP.NET_SessionId";

    private AtomicReference<SessionCookie> sessionCookie = new AtomicReference<>(null);
    private JAXBContext modelJC;
    private JAXBContext responsesJC;
    private MessageFactory messageFactory;
    private SparkClientProperties properties;


    @SneakyThrows
    public SparkClient(AsyncHttpClientWrapper asyncHttpClient, SparkClientProperties properties) {
        super(asyncHttpClient, properties, null);
        modelJC = JAXBContext.newInstance("ru.yandex.travel.integration.spark.model");
        responsesJC = JAXBContext.newInstance("ru.yandex.travel.integration.spark.responses");
        responsesJC = JAXBContext.newInstance(CompanyShortReportResponse.class, EntrepreneurShortReportResponse.class);
        messageFactory = MessageFactory.newInstance(SOAPConstants.SOAP_1_2_PROTOCOL);
        this.properties = properties;
    }

    private <Req, Rsp> CompletableFuture<Rsp> call(Req request, Class<Rsp> responseType, String purpose) {
        return authorize().thenCompose(theVoid -> {
            try {
                return sendRequest("POST", "", createSOAPRequest(request), responseType, purpose);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    public EntrepreneurShortReportResponse getEntrepreneurShortReportSync(String inn) {
        return sync(getEntrepreneurShortReport(inn));
    }

    public CompletableFuture<EntrepreneurShortReportResponse> getEntrepreneurShortReport(String inn) {
        GetEntrepreneurShortReport request = new GetEntrepreneurShortReport();
        request.setInn(inn);
        return call(request, GetEntrepreneurShortReportResponse.class, "GetEntrepreneurShortReport").thenApply(response -> {
            Preconditions.checkState(response.getGetEntrepreneurShortReportResult().equals("True"));
            return parseResponse(response.getXmlData(), EntrepreneurShortReportResponse.class);
        });
    }

    public CompanyShortReportResponse getCompanyShortReportSync(String inn) {
        return sync(getCompanyShortReport(inn));
    }

    public CompletableFuture<CompanyShortReportResponse> getCompanyShortReport(String inn) {
        GetCompanyShortReport request = new GetCompanyShortReport();
        request.setInn(inn);
        return call(request, GetCompanyShortReportResponse.class, "GetCompanyShortReport").thenApply(response -> {
            Preconditions.checkState(response.getGetCompanyShortReportResult().equals("True"));
            return parseResponse(response.getXmlData(), CompanyShortReportResponse.class);
        });
    }

    private CompletableFuture<Void> authorize() {
        SessionCookie cookie = sessionCookie.get();
        if (cookie != null && cookie.isValid(properties.getSessionTtl())) {
            log.info("Cookie is valid, using existing one. Now: {}. Expiration moment: {}",
                    Instant.now(),
                    cookie.createdAt.plus(properties.getSessionTtl()));
            return CompletableFuture.completedFuture(null);
        } else {
            if (cookie == null) {
                log.info("Cookie is not found. Authorizing");
            } else {
                log.info("Cookie expired, authorizing. Now: {}. Expiration moment: {}",
                        Instant.now(),
                        cookie.createdAt.plus(properties.getSessionTtl()));
            }
        }
        Authmethod authmethod = new Authmethod();
        authmethod.setLogin(properties.getLogin());
        authmethod.setPassword(properties.getPassword());
        SOAPMessage message = createSOAPRequest(authmethod);
        return sendRequest("POST", "", message, AuthmethodResponse.class, "auth").thenApply(response -> {
            Preconditions.checkState(response.getAuthmethodResult().equals("True"), "Failed to authorize");
            return null;
        });
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        rb.addHeader("Content-Type", "application/soap+xml; charset=utf-8");
        SessionCookie cookie = sessionCookie.get();
        if (cookie != null && cookie.isValid(properties.getSessionTtl())) {
            rb.addHeader(SESSION_HEADER_NAME, cookie.getSessionId());
        }
        return rb;
    }

    @Override
    protected <T> String serializeRequest(T request) {
        if (request == null) {
            return null;
        }
        if (!(request instanceof SOAPMessage)) {
            throw new RuntimeException("Unexpected message class");
        }
        SOAPMessage message = (SOAPMessage) request;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            message.writeTo(baos);
            return baos.toString();
        } catch (SOAPException | IOException e) {
            log.error("Error serializing request", e);
            throw new RuntimeException(e);
        }
    }

    private  <T> T parseResponse(String xmlData, Class<T> responseType) {
        try {
            Unmarshaller unmarshaller = responsesJC.createUnmarshaller();
            return unmarshaller.unmarshal(new StreamSource(new StringReader(xmlData)), responseType).getValue();
        } catch (JAXBException e) {
            log.error("Unable to parse to type: {}; xmlData: {} ", responseType, xmlData, e);
            throw new RuntimeException("Unable to parse xmlData", e);
        }
    }

    @Override
    protected <T> T parseSuccessfulContent(Response response, Class<T> contentType) throws Exception {
        if (Strings.isNullOrEmpty(response.getResponseBody()) || contentType == null) {
            return null;
        }
        SOAPMessage message = messageFactory.createMessage(
                new MimeHeaders(),
                new ByteArrayInputStream(response.getResponseBody().getBytes(StandardCharsets.UTF_8))
        );
        Unmarshaller unmarshaller = modelJC.createUnmarshaller();
        T result = unmarshaller.unmarshal(message.getSOAPBody().extractContentAsDocument(), contentType).getValue();
        if (result instanceof AuthmethodResponse) {
            manageSession(response, (AuthmethodResponse) result);
        }
        return result;
    }

    private void manageSession(Response response, AuthmethodResponse authResponse) {
        String cookieHeader = response.getHeaders().get("Set-Cookie");
        if (StringUtils.isNotBlank(cookieHeader)) {
            for (HttpCookie cookie : HttpCookie.parse(cookieHeader)) {
                if (cookie.getName().equals(SESSION_HEADER_NAME)) {
                    log.info("Cookie is refreshed. Now: {}. Expiration moment: {}",
                            Instant.now(),
                            Instant.now().plus(properties.getSessionTtl()));
                    sessionCookie.set(new SessionCookie(cookie.getValue(), Instant.now()));
                    return;
                }
            }
        }
        if (authResponse.getAuthmethodResult().equals("True")) {
            Preconditions.checkNotNull(sessionCookie.get(), "SessionCookie should be present already");
            log.info("Cookie's creation time is refreshed. Now: {}. Expiration moment: {}",
                    Instant.now(),
                    Instant.now().plus(properties.getSessionTtl()));
            sessionCookie.set(new SessionCookie(sessionCookie.get().sessionId, Instant.now()));
        }
    }

    @SneakyThrows
    private Document marshalToDocument(Object request) {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document document = db.newDocument();
        Marshaller marshaller = modelJC.createMarshaller();
        marshaller.marshal(request, document);
        return document;
    }

    @SneakyThrows
    private SOAPMessage createSOAPRequest(Object request) {
        SOAPMessage soapMessage = messageFactory.createMessage();
        createSoapEnvelope(soapMessage, request);
        soapMessage.saveChanges();
        return soapMessage;
    }

    @SneakyThrows
    private void createSoapEnvelope(SOAPMessage soapMessage, Object request) {
        SOAPPart soapPart = soapMessage.getSOAPPart();

        String myNamespace = "soap12";
        String myNamespaceURI = "http://www.w3.org/2003/05/soap-envelope";

        // SOAP Envelope
        SOAPEnvelope envelope = soapPart.getEnvelope();
        envelope.addNamespaceDeclaration(myNamespace, myNamespaceURI);

        // SOAP Body
        SOAPBody soapBody = envelope.getBody();
        soapBody.addDocument(marshalToDocument(request));
    }

    @RequiredArgsConstructor
    @Getter
    private static class SessionCookie {
        private final String sessionId;
        private final Instant createdAt;

        boolean isValid(Duration sessionTtl) {
            return createdAt.plus(sessionTtl).isAfter(Instant.now());
        }
    }
}
