package ru.yandex.travel.actuator;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import io.micrometer.core.instrument.Tag;

class MaskedTags implements Iterable<Tag> {
    private final List<Tag> tags;
    private final int mask;

    public MaskedTags(List<Tag> tags, int mask) {
        this.tags = tags;
        this.mask = mask;
    }

    static int nextIndex(int size, int mask, int index) {
        while (index < size && (mask & (1 << index)) == 0) {
            ++index;
        }
        return index;
    }

    int size() {
        return Integer.bitCount(mask);
    }

    @Override
    public Iterator<Tag> iterator() {
        return new Iterator<Tag>() {
            int index = nextIndex(tags.size(), mask, 0);

            @Override
            public boolean hasNext() {
                return index < tags.size();
            }

            @Override
            public Tag next() {
                int resultIndex = index;
                index = nextIndex(tags.size(), mask, index + 1);
                return tags.get(resultIndex);
            }
        };
    }

    String[] toArray() {
        ArrayList<String> items = new ArrayList<>();
        for (Tag tag : this) {
            items.add(tag.getKey());
            items.add(tag.getValue());
        }
        String[] array = new String[items.size()];
        return items.toArray(array);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MaskedTags that = (MaskedTags) o;
        int thisSize = this.tags.size();
        int thatSize = that.tags.size();
        int i = nextIndex(thisSize, this.mask, 0);
        int j = nextIndex(thatSize, that.mask, 0);
        while (true) {
            if (i < thisSize && j < thatSize) {
                Tag thisTag = this.tags.get(i);
                Tag thatTag = that.tags.get(j);
                if (!Objects.equals(thisTag.getKey(), thatTag.getKey())) {
                    return false;
                }
                if (!Objects.equals(thisTag.getValue(), thatTag.getValue())) {
                    return false;
                }
                i = nextIndex(thisSize, this.mask, i + 1);
                j = nextIndex(thatSize, that.mask, j + 1);
            } else {
                return i == thisSize && j == thatSize;
            }
        }
    }

    @Override
    public int hashCode() {
        int result = 17;
        for (int i = 0; i < tags.size(); ++i) {
            if ((mask & (1 << i)) != 0) {
                result = 31 * result ^ tags.get(i).hashCode();
            }
        }
        return result;
    }
}
