package ru.yandex.travel.credentials;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import com.google.common.base.Preconditions;
import io.grpc.Context;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.http.CommonHttpHeaders;

/**
 * This filter must be registered manually if the application needs to extract userCredentials info from
 * CommonHttpHeaders
 */
@Slf4j
@RequiredArgsConstructor
public class UserCredentialsServletFilter implements Filter {
    private final UserCredentialsBuilder userCredentialsBuilder;
    private final UserCredentialsPassportExtractor credentialsPassportExtractor;

    @Override
    public void init(FilterConfig filterConfig) {
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
            ServletException {
        CommonHttpHeaders httpHeaders = CommonHttpHeaders.get();
        Preconditions.checkNotNull(httpHeaders, "Cannot get common HTTP headers");
        String userTicketValue = httpHeaders.getUserTicket();
        String passportId = httpHeaders.getPassportId();
        passportId = credentialsPassportExtractor.extractPassportId(passportId, userTicketValue);

        UserCredentials userCredentials = userCredentialsBuilder.build(httpHeaders.getSessionKey(),
                httpHeaders.getYandexUid(), passportId, httpHeaders.getUserLogin(),
                userTicketValue, httpHeaders.getUserIP(),
                httpHeaders.getUserIsStaffAsBool(),
                httpHeaders.getUserIsPlusAsBool()
        );
        try {
            Context.current().withValue(UserCredentials.KEY, userCredentials).call(
                    () -> {
                        chain.doFilter(request, response);
                        return null;
                    }
            );
        } catch (IOException | ServletException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void destroy() {
    }
}
