package ru.yandex.travel.grpc;

import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import io.grpc.Context;

public class AppCallIdGenerator {

    public static final Context.Key<AppCallIdGenerator> KEY = Context.key("ru.yandex.travel.grpc.AppCallIdGenerator");

    private final String seed;
    private final ConcurrentMap<String, UUID> uuids = new ConcurrentHashMap<>();

    private AppCallIdGenerator(String seed) {
        this.seed = Preconditions.checkNotNull(seed, "Provided seed must be not null");
    }

    public static AppCallIdGenerator newInstance() {
        return newInstance(UUID.randomUUID().toString());
    }

    public static AppCallIdGenerator newInstance(String seed) {
        return new AppCallIdGenerator(seed);
    }

    /**
     * Generates a string that can be passed as grpc app-call-id and used to deduplicate requests and cache response on
     * the grpc service side.
     * Subsequent calls of this method for the same AppCallIdGenerator instance with the same parameter must
     * generate the same result
     *
     * @param methodName
     * @return string identifier
     */
    public String generate(String methodName) {
        Preconditions.checkNotNull(methodName, "Provided method name must be not null");
        Hasher hasher = Hashing.sha256().newHasher();
        hasher.putString(seed, Charsets.UTF_8);
        hasher.putString(methodName, Charsets.UTF_8);
        return hasher.hash().toString();
    }

    /**
     * For that rare case when we need UUID (e.g. avia create order method)
     *
     * @param methodName
     * @return identifier (guaranteed to be UUID)
     */
    @Deprecated
    public String generateUUID(String methodName) {
        return uuids.computeIfAbsent(methodName, ignored -> UUID.randomUUID()).toString();
    }
}
