package ru.yandex.travel.grpc;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.PropertyUtils;
import ru.yandex.travel.credentials.UserCredentialsAuthValidator;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractor;
import ru.yandex.travel.credentials.UserCredentialsValidator;
import ru.yandex.travel.grpc.interceptors.DefaultGrpcServiceInterceptorConfigurer;
import ru.yandex.travel.grpc.interceptors.LoggingServerInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingServerInterceptorBlacklistMethodsSupplier;
import ru.yandex.travel.grpc.interceptors.TracerServerInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderServerInterceptor;
import ru.yandex.travel.grpc.interceptors.UserCredentialsServerInterceptor;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@ConditionalOnBean(annotation = GrpcService.class)
@ConditionalOnProperty(prefix = "grpc", name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties({GrpcServerProperties.class, GrpcTvmProperties.class})
@RequiredArgsConstructor
public class GrpcServerAutoConfiguration {

    private final GrpcServerProperties grpcProperties;
    private final GrpcTvmProperties grpcTvmProperties;


    @Bean
    public GrpcServerRunner grpcServerRunner() {
        return new GrpcServerRunner(grpcProperties);
    }

    // Users can redefine this supplier in their apps and blacklist logging of some grpc methods
    @Bean
    @ConditionalOnMissingBean(LoggingServerInterceptorBlacklistMethodsSupplier.class)
    public LoggingServerInterceptorBlacklistMethodsSupplier loggingServerInterceptorBlacklistMethodsSupplier() {
        return Collections::emptySet;
    }

    @Bean
    public LoggingServerInterceptor loggingInterceptor(LoggingServerInterceptorBlacklistMethodsSupplier loggingServerInterceptorBlacklistMethodsSupplier,
                                                       ObjectProvider<Tracer> tracerObjectProvider) {
        try {
            String serverFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
            return new LoggingServerInterceptor(serverFqdn,
                    loggingServerInterceptorBlacklistMethodsSupplier.getBlacklistMethods(),
                    tracerObjectProvider.getIfAvailable(),
                    grpcProperties.isReportErrorsToSentry()
            );
        } catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    @Bean
    @ConditionalOnProperty(value = "grpc.tracing.enabled")
    public TracerServerInterceptor tracerServerInterceptor(Tracer tracer) {
        return new TracerServerInterceptor(tracer);
    }

    @Bean
    @ConditionalOnProperty(value = "grpc-tvm.enabled")
    @ConditionalOnMissingBean(TvmHeaderServerInterceptor.class)
    public TvmHeaderServerInterceptor tvmHeaderInterceptor(TvmWrapper tvm2) {
        List<String> allowedConsumers = PropertyUtils.stringToListOfStrings(grpcTvmProperties.getAllowedConsumers());
        return new TvmHeaderServerInterceptor(tvm2, allowedConsumers);
    }

    @Bean
    @ConditionalOnProperty(value = "credentials.enabled")
    @ConditionalOnMissingBean(UserCredentialsServerInterceptor.class)
    public UserCredentialsServerInterceptor userCredentialsServerInterceptor(
            UserCredentialsBuilder userCredentialsBuilder,
            UserCredentialsPassportExtractor userCredentialsPassportExtractor,
            UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsServerInterceptor(userCredentialsBuilder, userCredentialsPassportExtractor,
                new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Bean
    @ConditionalOnMissingBean(GrpcServiceInterceptorConfigurer.class)
    public DefaultGrpcServiceInterceptorConfigurer defaultGrpcServiceInterceptorConfigurer(
            LoggingServerInterceptor loggingServerInterceptor,
            @Autowired(required = false) TracerServerInterceptor tracerServerInterceptor,
            @Autowired(required = false) TvmHeaderServerInterceptor tvmHeaderServerInterceptor,
            @Autowired(required = false) UserCredentialsServerInterceptor userCredentialsServerInterceptor) {
        return new DefaultGrpcServiceInterceptorConfigurer(loggingServerInterceptor, tracerServerInterceptor,
                tvmHeaderServerInterceptor, userCredentialsServerInterceptor);
    }
}
