package ru.yandex.travel.grpc;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.BindableService;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.ServerInterceptor;
import io.grpc.ServerInterceptors;
import io.grpc.protobuf.services.ProtoReflectionService;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

public class GrpcServerRunner implements CommandLineRunner, DisposableBean, ApplicationContextAware {
    private static final Logger logger = LoggerFactory.getLogger(GrpcServerRunner.class);
    private final GrpcServerProperties grpcProperties;
    private ApplicationContext context;
    private Server server;

    public GrpcServerRunner(GrpcServerProperties grpcProperties) {
        this.grpcProperties = grpcProperties;
    }

    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        this.context = context;
    }

    @Override
    public void destroy() throws Exception {
        stop();
    }

    public void stop() throws InterruptedException {
        if (server == null) {
            return;
        }

        logger.info("Stopping gRPC server");

        context.publishEvent(new GrpcServerStoppedEvent(server));
        server.shutdown();
        server.awaitTermination();

        logger.info("gRPC server stopped");
        server = null;
    }

    @Override
    public void run(String... args) throws Exception {
        logger.info("Starting gRPC server");

        // TODO (mbobrov): use netty server builder here to fully customize it
        ServerBuilder builder = ServerBuilder.forPort(grpcProperties.getPort());
        if (grpcProperties.isEnableThreadPoolExecutor()) {
            ThreadFactoryBuilder threadFactoryBuilder = new ThreadFactoryBuilder();
            if (grpcProperties.getThreadPoolName() != null) {
                threadFactoryBuilder.setNameFormat(grpcProperties.getThreadPoolName());
            }
            threadFactoryBuilder.setDaemon(grpcProperties.isDaemonThreads());
            // TODO (mbobrov): move this executor to class member, so it can be stopped
            ExecutorService executorService = Executors.newFixedThreadPool(grpcProperties.getThreadPoolSize(),
                    threadFactoryBuilder.build());
            String threadPoolMetricName = "GrpcServiceExecutor";
            ExecutorServiceMetrics.monitor(Metrics.globalRegistry, executorService, threadPoolMetricName);
            builder.executor(executorService);
            logger.info("gRPC server will use separate thread pool of size " + grpcProperties.getThreadPoolSize());
        } else {
            builder.directExecutor();
        }

        context.getBeansWithAnnotation(GrpcService.class).forEach((s, o) -> {
            if (o instanceof BindableService) {
                BindableService service = ((BindableService) o);
                List<ServerInterceptor> interceptors = new ArrayList<>();

                context.getBeansOfType(GrpcServiceInterceptorConfigurer.class).forEach((configurerName, configurer) -> {
                    interceptors.addAll(configurer.getInterceptors(service));
                });
                builder.addService(ServerInterceptors.intercept(service, interceptors));
                logger.info("Registered service '{}'", service.getClass().getName());
            } else {
                throw new IllegalStateException(String.format("Service '%1$s' is not an instance of '%2$s'",
                        o.getClass().getName(),
                        BindableService.class.getName()));
            }
        });

        if (grpcProperties.isEnableReflectionService()) {
            builder.addService(ProtoReflectionService.newInstance());
        }

        server = builder.build();
        server.start();
        context.publishEvent(new GrpcServerInitializedEvent(server));

        logger.info("gRPC server started, listening on port {}", grpcProperties.getPort());
    }

}
