package ru.yandex.travel.grpc.interceptors;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Preconditions;
import io.grpc.BindableService;
import io.grpc.ServerInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.grpc.GrpcServiceInterceptorConfigurer;

public class DefaultGrpcServiceInterceptorConfigurer implements GrpcServiceInterceptorConfigurer {
    private static final Logger logger = LoggerFactory.getLogger(DefaultGrpcServiceInterceptorConfigurer.class);
    private final TracerServerInterceptor tracerServerInterceptor;
    private final LoggingServerInterceptor loggingInterceptor;
    private final TvmHeaderServerInterceptor tvmHeaderServerInterceptor;
    private final UserCredentialsServerInterceptor userCredentialsServerInterceptor;

    @Autowired
    public DefaultGrpcServiceInterceptorConfigurer(LoggingServerInterceptor loggingInterceptor,
                                                   @Autowired(required = false) TracerServerInterceptor tracerServerInterceptor,
                                                   @Autowired(required = false) TvmHeaderServerInterceptor tvmHeaderServerInterceptor,
                                                   @Autowired(required = false) UserCredentialsServerInterceptor userCredentialsServerInterceptor) {
        this.loggingInterceptor = loggingInterceptor;
        this.tracerServerInterceptor = tracerServerInterceptor;
        this.tvmHeaderServerInterceptor = tvmHeaderServerInterceptor;
        this.userCredentialsServerInterceptor = userCredentialsServerInterceptor;
    }

    @Override
    public List<ServerInterceptor> getInterceptors(BindableService service) {
        String serviceName = service.getClass().getName();
        List<ServerInterceptor> interceptors = new ArrayList<>();

        Class<?> originalClass = AopUtils.getTargetClass(service);
        GrpcService annotation = originalClass.getAnnotation(GrpcService.class);
        Preconditions.checkNotNull(annotation);
        if (annotation.authenticateUser()) {
            if (userCredentialsServerInterceptor != null) {
                interceptors.add(userCredentialsServerInterceptor);
                logger.info("Will enforce user authentication for service '{}'", serviceName);
            } else {
                logger.warn("Service '{}' requires user authentication, but appropriate interceptor is not " +
                        "configured", serviceName);
            }
        }
        if (annotation.authenticateService()) {
            if (tvmHeaderServerInterceptor != null) {
                interceptors.add(tvmHeaderServerInterceptor);
                logger.info("Will enforce cross-service authentication for service '{}'", serviceName);
            } else {
                logger.warn("Service '{}' requires cross-service authentication, but appropriate interceptor is not " +
                        "configured", serviceName);
            }
        }
        if (annotation.log()) {
            interceptors.add(loggingInterceptor);
        }
        if (annotation.trace()) {
            if (tracerServerInterceptor != null) {
                interceptors.add(tracerServerInterceptor);
                logger.info("Will trace service calls for service '{}'", serviceName);
            } else {
                logger.warn("Service '{}' requires tracing of calls, but appropriate interceptor is not configured",
                        serviceName);
            }
        }
        return interceptors;
    }
}
