package ru.yandex.travel.grpc.interceptors;

import com.google.common.base.Strings;
import io.grpc.Contexts;
import io.grpc.ForwardingServerCall;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.tag.Tags;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.tracing.GrpcMetadataExtractAdapter;

@Slf4j
public class TracerServerInterceptor implements ServerInterceptor {
    private final Tracer tracer;

    public TracerServerInterceptor(Tracer tracer) {
        this.tracer = tracer;
    }
    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call, Metadata headers, ServerCallHandler<ReqT, RespT> next) {

        String callId = headers.get(LoggingInterceptorCommons.METADATA_CALL_ID);

        if (Strings.isNullOrEmpty(callId)) {
            callId = "unknown";
        }

        SpanContext extractedContext = tracer.extract(Format.Builtin.HTTP_HEADERS,
                new GrpcMetadataExtractAdapter(headers));

        final Span span = tracer.buildSpan(call.getMethodDescriptor().getFullMethodName())
                .asChildOf(extractedContext)
                .withTag("CallId", callId)
                .withTag(Tags.SPAN_KIND.getKey(), Tags.SPAN_KIND_SERVER)
                .start();

        try (Scope scope = tracer.scopeManager().activate(span)) {
            ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT> interceptedCall =
                    new ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT>(call) {
                        @Override
                        public void sendHeaders(Metadata headers) {
                            super.sendHeaders(headers);
                        }

                        @Override
                        public void close(Status status, Metadata trailers) {
                            try(Scope scope = tracer.scopeManager().activate(span)) {
                                if (status.getCause() != null) {
                                    span.log("An excpetion has occured during the call. Cause: " + status.getCause().getMessage());
                                }
                                super.close(status, trailers);
                            } finally {
                                span.finish();
                            }
                        }
                    };
            return next.startCall(interceptedCall, headers);
        }
    }
}
