package ru.yandex.travel.grpc.interceptors;

import java.util.List;
import java.util.Objects;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.tvm.ServiceTicketCheck;
import ru.yandex.travel.tvm.TvmWrapper;


@Slf4j
public class TvmHeaderServerInterceptor implements ServerInterceptor {
    private final TvmWrapper tvm2;
    private final ImmutableSet<String> allowedConsumers;

    public TvmHeaderServerInterceptor(TvmWrapper tvm2, List<String> allowedConsumers) {
        Objects.requireNonNull(allowedConsumers, "Allowed src ids must be provided");
        this.tvm2 = tvm2;
        this.allowedConsumers = ImmutableSet.copyOf(allowedConsumers);
        tvm2.validateAliases(allowedConsumers);
    }

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call, Metadata headers, ServerCallHandler<ReqT, RespT> next) {
        String callId = headers.get(LoggingInterceptorCommons.METADATA_CALL_ID);

        if (Strings.isNullOrEmpty(callId)) {
            callId = "unknown";
        }

        log.debug("Checking TVM headers. (CallId: {})", callId);
        String serviceTicketHeader = headers.get(TvmHeaders.METADATA_SERVICE_TICKET_HEADER);
        if (serviceTicketHeader == null) {
            call.close(Status.UNAUTHENTICATED.withDescription("TVM service ticket not found. Please pass it using x-ya-service-ticket header"), headers);
            return GrpcInterceptorHelper.noOpListener();
        }

        log.debug("Validating TVM ticket. (CallId: {})", callId);
        ServiceTicketCheck ticketCheck = tvm2.checkServiceTicket(serviceTicketHeader, allowedConsumers);
        if (!ticketCheck.isOk()) {
            call.close(Status.UNAUTHENTICATED.withDescription(ticketCheck.getDescription()), headers);
            return GrpcInterceptorHelper.noOpListener();
        }

        log.debug("TVM ticket checked: (CallId: {})", callId);

        return next.startCall(call, headers);
    }


}
