package ru.yandex.travel.grpc.interceptors;

import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import lombok.RequiredArgsConstructor;

import ru.yandex.travel.credentials.BadUserCredentialsException;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractor;
import ru.yandex.travel.credentials.UserCredentialsValidator;

import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.LOGIN_HEADER;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.PASSPORT_ID_HEADER;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.SESSION_KEY_HEADER;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.USER_IP_HEADER;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.USER_IS_PLUS;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.USER_IS_STAFF;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.USER_TICKET_HEADER;
import static ru.yandex.travel.grpc.interceptors.UserCredentialsInterceptorCommons.YANDEX_UID_HEADER;

@RequiredArgsConstructor
public class UserCredentialsServerInterceptor implements ServerInterceptor {
    private final UserCredentialsBuilder userCredentialsBuilder;
    private final UserCredentialsPassportExtractor credentialsPassportExtractor;
    private final UserCredentialsValidator userCredentialsValidator;

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call, Metadata headers,
                                                                 ServerCallHandler<ReqT, RespT> next) {
        String userTicketValue = headers.get(USER_TICKET_HEADER);
        String yandexUid = headers.get(YANDEX_UID_HEADER);
        String login = headers.get(LOGIN_HEADER);
        String passportId = headers.get(PASSPORT_ID_HEADER);
        String sessionKey = headers.get(SESSION_KEY_HEADER);
        String userIp = headers.get(USER_IP_HEADER);
        boolean userIsStaff = headers.get(USER_IS_STAFF) != null && headers.get(USER_IS_STAFF);
        boolean userIsPlus = headers.get(USER_IS_PLUS) != null && headers.get(USER_IS_PLUS);
        passportId = credentialsPassportExtractor.extractPassportId(passportId, userTicketValue);
        UserCredentials userCredentials;
        try {
            userCredentials = userCredentialsBuilder.build(
                    sessionKey, yandexUid, passportId, login, userTicketValue, userIp,
                    userIsStaff, userIsPlus
            );
            userCredentialsValidator.validate(userCredentials);
        } catch (BadUserCredentialsException e) {
            call.close(Status.UNAUTHENTICATED.withDescription(e.getMessage()), headers);
            return GrpcInterceptorHelper.noOpListener();
        }
        Context context = Context.current().withValue(UserCredentials.KEY, userCredentials);
        return Contexts.interceptCall(context, call, headers, next);
    }
}
