package ru.yandex.travel.infrastructure;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationPreparedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.EnumerablePropertySource;
import org.springframework.core.env.MapPropertySource;

@Slf4j
public class ConfigProcessorApplicationListener implements ApplicationListener<ApplicationPreparedEvent> {

    @Override
    public void onApplicationEvent(ApplicationPreparedEvent event) {
        log.debug("ApplicationEnvironmentPreparedEvent fired, mapping secrets");
        ConfigurableEnvironment env = event.getApplicationContext().getEnvironment();

        boolean useMappingsFromFile = env.getProperty("properties-from-files.enabled", Boolean.class, false);
        if (!useMappingsFromFile) {
            log.debug("Skip mapping properties from files");
            return;
        }

        HashSet<String> autoMappingConfigProperties = new HashSet<>();
        env.getPropertySources().forEach(propertySource -> {
            if (propertySource instanceof EnumerablePropertySource && propertySource.getName().startsWith(
                    "applicationConfig")) {
                log.debug("Will map properties from source: " + propertySource.getName());
                autoMappingConfigProperties.addAll(Arrays
                        .stream(((EnumerablePropertySource) propertySource).getPropertyNames())
                        .filter(x -> x.startsWith("properties-from-files.mapping"))
                        .collect(Collectors.toList()));
            }
        });

        log.debug(String.format("Found auto mapping config properties: [%s]", String.join(", ",
                autoMappingConfigProperties)));

        HashMap<String, Object> autoMappedProperties = new HashMap<>();
        for (int i = 0; ; i++) {
            String nameKey = String.format("properties-from-files.mapping[%d].name", i);
            String pathKey = String.format("properties-from-files.mapping[%d].path", i);
            if (!autoMappingConfigProperties.contains(nameKey) || !autoMappingConfigProperties.contains(pathKey)) {
                break;
            }
            String name = env.getProperty(nameKey);
            String path = env.getProperty(pathKey);
            autoMappingConfigProperties.remove(nameKey);
            autoMappingConfigProperties.remove(pathKey);
            log.info("Mapping parameter '{}' value from path '{}'", name, path);
            String value;
            try {
                value = Files.readString(Paths.get(path)).strip();
            } catch (IOException e) {
                throw new IllegalStateException(
                        String.format("Failed to read auto-mapped property from file %s", path), e);
            }
            autoMappedProperties.put(name, value);
        }
        if (!autoMappingConfigProperties.isEmpty()) {
            throw new IllegalStateException(String.format("Found invalid auto mapping config properties: [%s]",
                    String.join(", ", autoMappingConfigProperties)));
        }

        env.getPropertySources().addFirst(new MapPropertySource("Travel properties properties-from-files",
                autoMappedProperties));
    }
}
