package ru.yandex.travel.ping;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import io.micrometer.core.instrument.Metrics;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthComponent;
import org.springframework.boot.actuate.health.HealthEndpoint;
import org.springframework.boot.actuate.health.Status;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("/ping")
public class PingController {
    private final HealthEndpoint healthEndpoint;
    private final PingControllerProperties properties;

    private AtomicLong lastBalancerPingForNannyMillis = new AtomicLong();
    private AtomicLong lastBalancerPingForAlertMillis = new AtomicLong();
    private AtomicLong firstSuccessfulPing = new AtomicLong();
    private AtomicBoolean warmupFinished = new AtomicBoolean();

    PingController(HealthEndpoint healthEndpoint, PingControllerProperties properties) {
        this.healthEndpoint = healthEndpoint;
        this.properties = properties;
        this.lastBalancerPingForNannyMillis.set(0);
        this.lastBalancerPingForAlertMillis.set(System.currentTimeMillis());// To avoid alert at startup
        this.firstSuccessfulPing.set(0);
        this.warmupFinished.set(false);
        Metrics.gauge("service.balancerPingIsMissing", this, PingController::getBalancerPingIsMissing);
    }

    @RequestMapping("/nanny")
    public ResponseEntity<String> nannyPing() {
        long now = System.currentTimeMillis();
        return makeResponse(getHealthForNannyResponse(now), now);
    }

    private HealthComponent getHealthForNannyResponse(long now) {
        if (properties.getBalancerPingPeriodForNanny() != null) {
            long lateness = now - lastBalancerPingForNannyMillis.get();
            if (lateness >= properties.getBalancerPingPeriodForNanny().toMillis()) {
                // When balancer doesn't ping application, Nanny should stop all reallocations
                return Health.down().withDetail("No balancer ping, last ping was (msec) ago", lateness).build();
            }
        }
        if (!warmupFinished.get()) {
            return Health.down().withDetail("Warm up", false).build();
        }
        return healthEndpoint.health();
    }

    @RequestMapping("/balancer")
    public ResponseEntity<String> balancerPing() {
        long now = System.currentTimeMillis();
        lastBalancerPingForNannyMillis.set(now);
        lastBalancerPingForAlertMillis.set(now);
        return makeResponse(healthEndpoint.health(), now);
    }

    private ResponseEntity<String> makeResponse(HealthComponent health, long now) {
        if (health.getStatus() == Status.UP) {
            firstSuccessfulPing.compareAndSet(0, now);
        }

        String weightHeader = null;
        boolean decreasedWeight = false;
        if (properties.getDecreasedWeightDuration() != null) {
            weightHeader = properties.getNormalWeightHeaderValue();
            if (firstSuccessfulPing.get() == 0 || now - firstSuccessfulPing.get() <= properties.getDecreasedWeightDuration().toMillis()) {
                weightHeader = properties.getDecreasedWeightHeaderValue();
                decreasedWeight = true;
            }
        }

        if (health.getStatus() != Status.UP) {
            return buildServiceUnavailableResponse(weightHeader, health.toString());
        }

        if (firstSuccessfulPing.get() == 0 || now - firstSuccessfulPing.get() <= properties.getHealthUpDelay().toMillis()) {
            return buildServiceUnavailableResponse(weightHeader, "Service is warming up");
        }

        if (decreasedWeight) {
            return buildServiceUpResponse(weightHeader, "Up (with decreased weight)");
        }
        warmupFinished.set(true);

        return buildServiceUpResponse(weightHeader, "Up");
    }

    private ResponseEntity<String> buildServiceUnavailableResponse(String weightHeader, String message) {
        var builder = ResponseEntity.status(HttpStatus.SERVICE_UNAVAILABLE).contentType(MediaType.TEXT_PLAIN);
        if (weightHeader != null) {
            builder.header("RS-Weight", weightHeader);
        }
        return builder.body(message);
    }

    private ResponseEntity<String> buildServiceUpResponse(String weightHeader, String message) {
        var builder = ResponseEntity.ok();
        if (weightHeader != null) {
            builder.header("RS-Weight", weightHeader);
        }
        return builder.body(message);
    }

    private double getBalancerPingIsMissing() {
        if (properties.getBalancerPingPeriodForAlert() == null) {
            return 0;
        }
        long lateness = System.currentTimeMillis() - lastBalancerPingForAlertMillis.get();
        if (lateness < properties.getBalancerPingPeriodForAlert().toMillis()) {
            return 0;
        } else {
            return 1;
        }
    }
}
