package ru.yandex.travel.task_processor;

import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;

/**
 * Base task key provider with built-in back pressure
 *
 * @param <K>
 */
@Slf4j
public abstract class AbstractTaskKeyProvider<K extends Comparable<K>> implements TaskKeyProvider<K> {

    private final ConcurrentSkipListSet<K> lockedTasks;

    public AbstractTaskKeyProvider() {
        this.lockedTasks = new ConcurrentSkipListSet<>();
    }

    protected Set<K> getLockedTaskKeys() {
        return ImmutableSet.copyOf(lockedTasks);
    }

    // an utility for jpa-based providers
    protected Set<K> getLockedTaskKeysSafe(Set<K> safeFakeKeys) {
        Set<K> lockedKeys = getLockedTaskKeys();
        return !lockedKeys.isEmpty() ? lockedKeys : safeFakeKeys;
    }

    @Override
    public boolean acquireTaskLock(K taskKey) {
        return lockedTasks.add(taskKey);
    }

    @Override
    public void releaseTaskLock(K taskKey) {
        Preconditions.checkArgument(lockedTasks.remove(taskKey),
                "Some huge problems with task processor locked keys: failed to remove the key %s from the set %s",
                    taskKey, lockedTasks);
    }
}
