package ru.yandex.travel.task_processor;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class PausableTerminationSemaphore {

    private String name;

    public enum State {
        PAUSED, ACTIVE, TERMINATING
    }

    private volatile State state = State.PAUSED;
    // volatile so we can safely read the latest value, all the changes to the variable are made in synchronized methods
    private volatile int permits = 0;

    private final int maxPermits;

    public PausableTerminationSemaphore(String name) {
        this(name, Integer.MAX_VALUE);
    }

    public PausableTerminationSemaphore(String name, int maxPermits) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(name), "Semaphore name must be provided");
        Preconditions.checkArgument(maxPermits > 0, "Max permits must be positive");
        this.name = name;
        this.maxPermits = maxPermits;
    }

    public synchronized void pause() throws InterruptedException {
        Preconditions.checkState(state == State.ACTIVE, "Semaphore can be paused only in ACTIVE state");
        log.debug("Pausing semaphore {}. Permits: {}", name, permits);
        state = State.PAUSED;
        while (permits > 0) {
            wait();
        }
    }

    public boolean isActive() {
        return state == State.ACTIVE;
    }

    public int getAvailablePermits() {
        return maxPermits - permits;
    }

    public int getPermits() {
        return permits;
    }

    public int getMaxPermits() {
        return maxPermits;
    }

    public synchronized void resume() {
        Preconditions.checkState(state == State.PAUSED, "Semaphore can be resumed only from PAUSED state");
        state = State.ACTIVE;
    }

    public synchronized boolean acquire() {
        if (state == State.ACTIVE && permits < maxPermits) {
            permits++;
            log.debug("Acquired permit from semaphore {}. Permits: {}", name, permits);
            return true;
        } else {
            log.debug("Couldn't acquire permit from semaphore {}. Permits: {}", name, permits);
            return false;
        }
    }

    public synchronized void shutdown() {
        log.debug("Shutting down {}. Permits: {}", name, permits);
        state = State.TERMINATING;
    }

    public synchronized void awaitTermination() throws InterruptedException {
        Preconditions.checkState(state == State.TERMINATING, "Awaiting termination for non terminated semaphore");
        log.debug("Awaiting termination {}. Permits: {}", name, permits);
        while (permits > 0) {
            wait();
        }
    }

    public synchronized void release() {
        Preconditions.checkState(permits > 0, "Trying to release semaphore that was not acquired");
        permits--;
        log.debug("Released permit {}. Permits: {}", name, permits);
        if (permits == 0 && (state == State.PAUSED || state == State.TERMINATING)) {
            notifyAll();
        }
    }
}
