package ru.yandex.travel.task_processor;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class PausableTerminationSemaphoreWithBuckets {

    private String name;

    private enum State {
        PAUSED, ACTIVE, TERMINATING
    }

    private volatile State state = State.PAUSED;
    private ImmutableMap<Integer, Permits> buckets;

    public PausableTerminationSemaphoreWithBuckets(String name, Map<Integer, Integer> maxBucketPermits) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(name), "Semaphore name must be provided");
        Preconditions.checkArgument(maxBucketPermits != null && !maxBucketPermits.isEmpty(), "No buckets with permits");
        for (Map.Entry<Integer, Integer> entry : maxBucketPermits.entrySet()) {
            Preconditions.checkArgument(entry.getValue() > 0, "Invalid permits for bucket %s: %s",
                    entry.getKey(), entry.getValue());
        }
        this.name = name;
        this.buckets = ImmutableMap.copyOf(maxBucketPermits.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> new Permits(e.getKey(), e.getValue()))));
    }

    public synchronized void pause() throws InterruptedException {
        Preconditions.checkState(state == State.ACTIVE, "Semaphore can be paused only in ACTIVE state");
        log.debug("Pausing semaphore {}. Permits: {}", name, buckets);
        state = State.PAUSED;
        while (getPermits() > 0) {
            wait();
        }
    }

    public boolean isActive() {
        return state == State.ACTIVE;
    }

    public int getMaxPermits() {
        return buckets.values().stream().mapToInt(Permits::getMax).sum();
    }

    public int getAvailablePermits() {
        return buckets.values().stream().mapToInt(Permits::getAvailable).sum();
    }

    public Map<Integer, Integer> getAvailableBucketPermits() {
        return buckets.values().stream().collect(Collectors.toMap(Permits::getBucketId, Permits::getAvailable));
    }

    public int getPermits() {
        return buckets.values().stream().mapToInt(Permits::getActive).sum();
    }

    public int getPermitsInUse(int bucketId) {
        return getBucketPermits(bucketId).getActive();
    }

    public synchronized void resume() {
        Preconditions.checkState(state == State.PAUSED, "Semaphore can be resumed only from PAUSED state");
        state = State.ACTIVE;
    }

    public boolean hasBucket(int bucketId) {
        return buckets.containsKey(bucketId);
    }

    private Permits getBucketPermits(int bucketId) {
        Permits permits = buckets.get(bucketId);
        Preconditions.checkArgument(permits != null, "No such bucket: %s", bucketId);
        return permits;
    }

    public synchronized boolean acquire(int bucketId) {
        Permits permits = getBucketPermits(bucketId);
        if (state == State.ACTIVE && permits.acquire()) {
            log.debug("Acquired permit from semaphore {}. Bucket: {}. Permits: {}", name, bucketId, permits);
            return true;
        } else {
            log.debug("Couldn't acquire permit from semaphore {}. Bucket: {}. Permits: {}", name, bucketId, permits);
            return false;
        }
    }

    public synchronized void shutdown() {
        log.debug("Shutting down {}. Permits: {}", name, buckets);
        state = State.TERMINATING;
    }

    public synchronized void awaitTermination() throws InterruptedException {
        Preconditions.checkState(state == State.TERMINATING, "Awaiting termination for non terminated semaphore");
        log.debug("Awaiting termination {}. Permits: {}", name, buckets);
        while (getPermits() > 0) {
            wait();
        }
    }

    public synchronized void release(int bucketId) {
        Permits permits = getBucketPermits(bucketId);
        permits.release();
        log.debug("Released permit {}. Bucket: {}. Permits: {}", name, bucketId, permits);
        if (getPermits() == 0 && (state == State.PAUSED || state == State.TERMINATING)) {
            notifyAll();
        }
    }

    @AllArgsConstructor
    private static class Permits {
        @Getter
        private final int bucketId;
        private final AtomicInteger active = new AtomicInteger(0);
        @Getter
        private final int max;

        public int getActive() {
            return active.get();
        }

        public int getAvailable() {
            return max - active.get();
        }

        // access to this method has to be synchronized externally
        public boolean acquire() {
            if (max > active.get()) {
                active.incrementAndGet();
                return true;
            } else {
                return false;
            }
        }

        // access to this method has to be synchronized externally
        public void release() {
            Preconditions.checkState(active.get() > 0,
                    "Trying to release semaphore that was not acquired: bucket %s", bucketId);
            active.decrementAndGet();
        }

        @Override
        public String toString() {
            return active + "/" + max;
        }
    }
}
