package ru.yandex.travel.orders.services.payments;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.IAsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.masking.LogAwareRequestBuilder;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustClearResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketPassParams;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustOrderStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsRequest;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.services.payments.model.TrustResizeResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustUnholdResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustGetAccountsResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStartResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStatusResponse;

@Slf4j
public class DefaultTrustClient implements TrustClient {
    private static final Set<Integer> SUCCESSFUL_RESPONSE_CODES = ImmutableSet.of(200, 201);

    private final IAsyncHttpClientWrapper asyncHttpClient;
    private final TrustConnectionProperties trustConfig;
    private final String serviceToken;
    private final ObjectMapper objectMapper;
    private final TrustCreateBasketPassParams passParams;

    public DefaultTrustClient(IAsyncHttpClientWrapper asyncHttpClient, TrustConnectionProperties trustConfig,
                              String serviceToken) {
        this.asyncHttpClient = asyncHttpClient;
        this.trustConfig = trustConfig;
        this.serviceToken = serviceToken;
        this.passParams = null;
        this.objectMapper = createJsonMapper();
    }

    public DefaultTrustClient(AsyncHttpClientWrapper asyncHttpClient, TrustConnectionProperties trustConfig,
                              String serviceToken, TrustCreateBasketPassParams passParams) {
        this.asyncHttpClient = asyncHttpClient;
        this.trustConfig = trustConfig;
        this.serviceToken = serviceToken;
        this.passParams = passParams;
        this.objectMapper = createJsonMapper();
    }

    public static ObjectMapper createJsonMapper() {
        return new ObjectMapper()
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .registerModule(new JavaTimeModule());
    }

    @Override
    public TrustCreateOrderResponse createOrder(String productId, TrustUserInfo userInfo) {
        TrustCreateOrderRequest body = new TrustCreateOrderRequest();
        body.setProductId(productId);
        return sync(sendRequest(
                "POST", "/orders", userInfo, body, TrustCreateOrderResponse.class, Method.CREATE_ORDER.toString()
        ));
    }

    //@Override
    public TrustOrderStatusResponse getOrder(String orderId, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "GET", "/orders/" + orderId, userInfo, null, TrustOrderStatusResponse.class, Method.GET_ORDER.toString()
        ));
    }

    @Override
    public TrustCreateBasketResponse createBasket(TrustCreateBasketRequest request, TrustUserInfo userInfo,
                                                  Object testContext) {
        Preconditions.checkArgument(testContext == null, "Test context should be disabled with actual trust");
        //TODO(mbobrov) TRAVELBACK-1170: remove this code after prod migration
        if (passParams != null && request.getPassParams() == null) {
            request.setPassParams(passParams);

        }
        return sync(sendRequest(
                "POST", "/payments", userInfo, request, TrustCreateBasketResponse.class, Method.CREATE_BASKET.toString()
        ));
    }

    @Override
    public TrustBasketStatusResponse getBasketStatus(String purchaseToken, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "GET", "/payments/" + purchaseToken + "?show_trust_payment_id=1&with_terminal_info=1", userInfo, null,
                TrustBasketStatusResponse.class,
                Method.GET_BASKET_STATUS.toString()
        ));
    }

    @Override
    public TrustStartPaymentResponse startPayment(String purchaseToken, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/payments/" + purchaseToken + "/start", userInfo, null, TrustStartPaymentResponse.class,
                Method.START_PAYMENT.toString()
        ));
    }

    @Override
    public TrustClearResponse clear(String purchaseToken, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/payments/" + purchaseToken + "/clear", userInfo, null, TrustClearResponse.class,
                Method.CLEAR.toString()
        ));
    }

    @Override
    public TrustUnholdResponse unhold(String purchaseToken, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/payments/" + purchaseToken + "/unhold", userInfo, null, TrustUnholdResponse.class,
                Method.CANCEL.toString()
        ));
    }

    @Override
    public TrustResizeResponse resize(String purchaseToken, String orderId, TrustResizeRequest resizeRequest,
                                      TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/payments/" + purchaseToken + "/orders/" + orderId + "/resize",
                userInfo, resizeRequest, TrustResizeResponse.class, Method.RESIZE.toString()
        ));
    }

    @Override
    public TrustCreateRefundResponse createRefund(TrustCreateRefundRequest request, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/refunds", userInfo,
                request, TrustCreateRefundResponse.class, Method.CREATE_REFUND.toString()
        ));
    }

    @Override
    public TrustStartRefundResponse startRefund(String refundId, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/refunds/" + refundId + "/start", userInfo, null, TrustStartRefundResponse.class,
                Method.START_REFUND.toString()
        ));
    }

    @Override
    public TrustRefundStatusResponse getRefundStatus(String refundId, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "GET", "/refunds/" + refundId, userInfo, null, TrustRefundStatusResponse.class,
                Method.GET_REFUND_STATUS.toString()
        ));
    }

    @Override
    public TrustPaymentMethodsResponse getPaymentMethods(TrustUserInfo userInfo) {
        return getPaymentMethods(null, userInfo);
    }

    public TrustGetAccountsResponse getAccounts(TrustUserInfo userInfo) {
        return sync(sendRequest(
                "GET", "/account/", userInfo, null, TrustGetAccountsResponse.class,
                Method.GET_ACCOUNT.toString()
        ));
    }

    @Override
    public TrustCreateAccountResponse createAccount(TrustCreateAccountRequest request, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "POST", "/account/", userInfo, request, TrustCreateAccountResponse.class,
                Method.CREATE_ACCOUNT.toString()
        ));
    }

    @Override
    public TrustCreateTopupResponse createTopup(TrustTopupRequest request, TrustUserInfo userInfo) {
        return syncPost("/topup", userInfo, request, TrustCreateTopupResponse.class, Method.CREATE_TOPUP.name());
    }

    @Override
    public TrustTopupStatusResponse getTopupStatus(String purchaseToken, TrustUserInfo userInfo) {
        return syncGet("/topup/" + purchaseToken,
                userInfo,
                TrustTopupStatusResponse.class,
                Method.GET_TOPUP_STATUS.name());
    }

    @Override
    public TrustTopupStartResponse startTopup(String purchaseToken, TrustUserInfo userInfo) {
        return syncPost("/topup/" + purchaseToken + "/start",
                userInfo,
                null,
                TrustTopupStartResponse.class,
                Method.START_TOPUP.name());
    }

    @Override
    public TrustPaymentReceiptResponse getReceipt(String purchaseToken, String receiptId, TrustUserInfo userInfo) {
        return sync(sendRequest(
                "GET",
                "/payments/" + purchaseToken + "/receipts/" + receiptId,
                userInfo,
                null,
                TrustPaymentReceiptResponse.class,
                Method.GET_RECEIPT.toString()
        ));
    }

    // todo(tlg-13) move to the interface
    //@Override
    public TrustPaymentMethodsResponse getPaymentMethods(TrustPaymentMethodsRequest request, TrustUserInfo userInfo) {
        return sync(getPaymentMethodsAsync(request, userInfo));
    }

    public CompletableFuture<TrustPaymentMethodsResponse> getPaymentMethodsAsync(TrustPaymentMethodsRequest request,
                                                                                 TrustUserInfo userInfo) {
        return sendRequest("GET", "/payment-methods", userInfo, request,
                TrustPaymentMethodsResponse.class, Method.GET_PAYMENT_METHODS.toString());
    }

    public CompletableFuture<TrustPaymentMethodsResponse> getPaymentMethodsAsync(TrustUserInfo userInfo) {
        return getPaymentMethodsAsync(null, userInfo);
    }

    private <T> T syncPost(String url,
                           TrustUserInfo userInfo,
                           Object request,
                           Class<T> responseClass,
                           String purpose) {
        return sync(sendRequest(
                "POST", url, userInfo, request, responseClass, purpose
        ));
    }

    private <T> T syncGet(String url,
                          TrustUserInfo userInfo,
                          Class<T> responseClass,
                          String purpose) {
        return sync(sendRequest(
                "GET", url, userInfo, null, responseClass, purpose
        ));
    }

    private <RQ, RS> CompletableFuture<RS> sendRequest(String method, String path, TrustUserInfo userInfo, RQ body,
                                                       Class<RS> responseType, String purpose) {
        String bodyString = serializeRequest(body);
        var requestBuilder = createBaseRequestBuilder(method, userInfo)
                .setUrl(trustConfig.getBaseUrl() + path)
                .setBody(body != null ? bodyString : null);
        return asyncHttpClient.executeRequest((LogAwareRequestBuilder) requestBuilder, purpose, null,
                (IAsyncHttpClientWrapper.ResponseParser<RS>) r -> parseResponse(r, responseType)
        );
    }

    private <T> T parseResponse(Response response, Class<T> resultClass) {
        if (SUCCESSFUL_RESPONSE_CODES.contains(response.getStatusCode())) {
            try {
                return objectMapper.readValue(response.getResponseBody(), resultClass);
            } catch (IOException e) {
                log.error("Unable to parse response", e);
                throw new RuntimeException("Unable to parse response", e);
            }
        } else if (response.getStatusCode() == 429) {
            throw new HttpApiRetryableException("Got response from trust with status code: " + response.getStatusCode());
        } else if (response.getStatusCode() >= 400 && response.getStatusCode() < 500) {
            throw new RuntimeException("Got response from trust with status code: " + response.getStatusCode() +
                    ", data: " + response.getResponseBody());
        } else if (response.getStatusCode() >= 500) {
            throw new HttpApiRetryableException("Got response from trust with status code: " + response.getStatusCode());
        } else {
            throw new RuntimeException(MessageFormat.format("Do not know how to handle status {0} from trust ({1})",
                    response.getStatusCode(), response.getResponseBody()));
        }
    }

    private <T> String serializeRequest(T request) {
        try {
            return objectMapper.writeValueAsString(request);
        } catch (JsonProcessingException e) {
            log.error("Error serializing request", e);
            throw new RuntimeException(e);
        }
    }

    private LogAwareRequestBuilder createBaseRequestBuilder(String method, TrustUserInfo userInfo) {
        var builder = new LogAwareRequestBuilder()
                .setHeader("X-Service-Token", serviceToken)
                .setReadTimeout(Math.toIntExact(trustConfig.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(trustConfig.getHttpRequestTimeout().toMillis()))
                .setMethod(method);
        if (userInfo != null) {
            if (!Strings.isNullOrEmpty(userInfo.getUid())) {
                builder.setHeader("X-Uid", userInfo.getUid());
            }
            if (!Strings.isNullOrEmpty(userInfo.getUserIp())) {
                builder.setHeader("X-User-Ip", userInfo.getUserIp());
            }
        }
        return (LogAwareRequestBuilder) builder;
    }

    private static <T> T sync(CompletableFuture<T> future) {
        try {
            return future.get();
        } catch (InterruptedException e) {
            log.error("Trust call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new HttpApiRetryableException(e.getMessage());
        } catch (ExecutionException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            }
            Throwable cause = e.getCause();
            if (cause instanceof TimeoutException) {
                throw new HttpApiRetryableException(e.getMessage());
            } else if (cause instanceof IOException) {
                throw new HttpApiRetryableException(e.getMessage());
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e);
            }
        }
    }

    public enum Method {
        CREATE_ORDER,
        GET_ORDER,
        CREATE_BASKET,
        GET_BASKET_STATUS,
        START_PAYMENT,
        CLEAR,
        CANCEL,
        RESIZE,
        CREATE_REFUND,
        START_REFUND,
        GET_REFUND_STATUS,
        GET_PAYMENT_METHODS,
        GET_RECEIPT,
        // yandex-plus methods
        GET_ACCOUNT,
        CREATE_ACCOUNT,
        CREATE_TOPUP,
        GET_TOPUP_STATUS,
        START_TOPUP,
        ;

        public static Set<String> getNames() {
            return Arrays.stream(Method.values()).map(Enum::toString).collect(Collectors.toSet());
        }
    }
}
