package ru.yandex.travel.orders.services.payments.model;

import java.math.BigDecimal;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.Builder;
import lombok.Value;
import lombok.extern.jackson.Jacksonized;

import ru.yandex.travel.commons.lang.ComparatorUtils;

/**
 * Details on composite payments:
 * <ul>
 * <li>https://wiki.yandex-team.ru/trust/yandexaccount/#chastichnajaoplatakorzinybonusnymiballami</li>
 * <li>https://wiki.yandex-team.ru/trust/composite-payments/#sozdaniekorzinysrazmetkojj</li>
 * </ul>
 *
 * <p/>For refunds, don't specify 0 sums. Skip them instead or they will cause strange 400 errors.
 */
@Value
@Jacksonized
@Builder
@JsonInclude(JsonInclude.Include.NON_NULL)
public final class TrustCompositeOrderPaymentMarkup {
    // each field corresponds to a certain TrustBoundPaymentMethodType
    @JsonSerialize(using = AmountSerializer.class)
    private final BigDecimal card;
    @JsonSerialize(using = AmountSerializer.class)
    private final BigDecimal phone;
    @JsonSerialize(using = AmountSerializer.class)
    private final BigDecimal yandexMoney;
    @JsonSerialize(using = AmountSerializer.class)
    private final BigDecimal yandexAccount;

    public TrustCompositeOrderPaymentMarkup withoutZeros() {
        return TrustCompositeOrderPaymentMarkup.builder()
                .card(zeroToNull(card))
                .phone(zeroToNull(phone))
                .yandexMoney(zeroToNull(yandexMoney))
                .yandexAccount(zeroToNull(yandexAccount))
                .build();
    }

    private BigDecimal zeroToNull(BigDecimal value) {
        return value != null && !ComparatorUtils.isEqualByCompareTo(value, BigDecimal.ZERO) ? value : null;
    }
}
