package ru.yandex.travel.tvm;

import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.LongStream;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.BiMap;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;
import ru.yandex.passport.tvmauth.TicketStatus;

@RequiredArgsConstructor
@Slf4j
public class TvmWrapperImpl implements TvmWrapper {
    private final Tvm2 tvm;
    private final BiMap<String, Integer> aliases;
    @Getter
    private final int selfTvmId;

    @Override
    public void validateAlias(String alias) {
        if (!this.aliases.containsKey(alias)) {
            throw new IllegalArgumentException("Unknown service alias: " + alias);
        }
    }

    @Override
    public void validateAliases(Collection<String> aliases) {
        Set<String> unknown = new TreeSet<>();
        for (String alias : aliases) {
            if (!this.aliases.containsKey(alias)) {
                unknown.add(alias);
            }
        }
        if (!unknown.isEmpty()) {
            throw new IllegalArgumentException("Unknown service aliases: " + unknown);
        }
    }

    @Override
    public String getServiceTicket(String alias) {
        Integer tvmId = aliases.get(alias);
        Preconditions.checkNotNull(tvmId, "Unknown tvm alias: %s", alias);
        return tvm.getServiceTicket(tvmId)
                .orElseThrow(() -> new NoSuchElementException(
                        "Failed to get a service ticket for the destination: alias=" + alias));
    }

    @Override
    public Optional<String> getServiceTicketOptional(String alias) {
        Integer tvmId = aliases.get(alias);
        Preconditions.checkNotNull(tvmId, "Unknown tvm alias: %s", alias);
        return Optional.ofNullable(tvm.getServiceTicket(tvmId).orElse((String) null));
    }

    @Override
    public ServiceTicketCheck checkServiceTicket(String serviceTicket, Collection<String> allowedSources) {
        if (Strings.isNullOrEmpty(serviceTicket)) {
            return ServiceTicketCheck.invalid(ServiceTicketCheckStatus.MISSING_TICKET,
                    "No service ticket provided");
        }

        CheckedServiceTicket ticket;
        try {
            ticket = tvm.checkServiceTicket(serviceTicket);
        } catch (Exception e) {
            log.warn("Unexpected error during tvm.checkServiceTicket", e);
            return ServiceTicketCheck.invalid(ServiceTicketCheckStatus.UNEXPECTED_ERROR,
                    "An error occurred while checking service ticket: " + e.getMessage());
        }

        if (ticket.getStatus() != ru.yandex.passport.tvmauth.TicketStatus.OK) {
            return ServiceTicketCheck.invalid(ServiceTicketCheckStatus.INVALID_TICKET,
                    "Service ticket status: " + ticket.getStatus());
        }

        String srcAlias = aliases.inverse().get(ticket.getSrc());
        if (!allowedSources.contains(srcAlias)) {
            return ServiceTicketCheck.invalid(ServiceTicketCheckStatus.NOT_ALLOWED_SOURCE,
                    "ServiceTicket comes from an unauthorized src: service_id=" + ticket.getSrc() + ", alias=" + srcAlias);
        }

        return ServiceTicketCheck.VALID;
    }


    @Override
    public UserTicketCheck checkUserTicket(String userTicket, Long passportId) {
        if (Strings.isNullOrEmpty(userTicket)) {
            return new UserTicketCheck(UserTicketCheckStatus.MISSING_TICKET, "No user ticket provided", null);
        }

        CheckedUserTicket ticket = tvm.checkUserTicket(userTicket);
        if (ticket.getStatus() != TicketStatus.OK) {
            return new UserTicketCheck(UserTicketCheckStatus.INVALID_TICKET, "Invalid user ticket", null);
        }
        if (passportId != null && LongStream.of(ticket.getUids()).noneMatch(uid -> uid == passportId)) {
            return new UserTicketCheck(UserTicketCheckStatus.PASSPORT_ID_MISMATCH, "Passport id does not match the ticket", null);
        }
        return new UserTicketCheck(UserTicketCheckStatus.OK, null, ticket.getDefaultUid());
    }

    @Override
    public void close() {
        tvm.stop();
    }

    @Override
    public boolean isInitialized() {
        return tvm.isInitialized();
    }
}
