package ru.yandex.travel.workflow;

import java.util.LinkedList;
import java.util.List;

import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.protobuf.util.JsonFormat;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

// Utility class to register logging events after transaction commit
@Slf4j
public class AfterCommitEventLogging {

    private static final class LoggerAndEventsHolder {
        private final Logger logger;

        private final List<LoggedEventInfo> loggedEventInfos;

        public LoggerAndEventsHolder(Logger logger) {
            this.logger = logger;
            this.loggedEventInfos = new LinkedList<>();
        }

        private void logAfterTransaction(Message event, boolean printDefaultValues) {
            loggedEventInfos.add(new LoggedEventInfo(event, printDefaultValues));
        }
    }

    @RequiredArgsConstructor
    private static final class LoggedEventInfo {
        private final Message event;
        private final boolean printDefaultValues;
    }


    /**
     * Append event to a list of events that'll be logged after transaction commit
     */
    public static void logEvent(Logger logger, Message event) {
        logEvent(logger, event, false);
    }

    public static void logEvent(Logger logger, Message event, boolean printDefaultValues) {
        innerLogEvent(logger, event, true, printDefaultValues);
    }

    /**
     * Convenience method not failing logging if transaction is not active
     * TODO (mbobrov): think of deprecating it, but now it's used for Workflow.createWorkflowForEntity method
     */
    public static void tryLogEvent(Logger logger, Message event) {
        innerLogEvent(logger, event, false, false);
    }

    private static void innerLogEvent(Logger logger, Message event, boolean activeTransactionRequired,
                                      boolean printDefaultValues) {
        Preconditions.checkNotNull(logger, "Logger must be provided");
        LoggerAndEventsHolder result =
                (LoggerAndEventsHolder) TransactionSynchronizationManager.getResource(resourceKey(logger));
        if (result == null) {
            if (TransactionSynchronizationManager.isSynchronizationActive()) {
                result = new LoggerAndEventsHolder(logger);
                TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization(result));
                TransactionSynchronizationManager.bindResource(resourceKey(logger), result);
                result.logAfterTransaction(event, printDefaultValues);
            } else {
                Preconditions.checkState(!activeTransactionRequired,
                        "Registering a logger must be done in transaction");
                doLogEvent(logger, event, printDefaultValues);
            }
        } else {
            result.logAfterTransaction(event, printDefaultValues);
        }

    }

    @RequiredArgsConstructor
    public static final class TransactionSynchronization extends TransactionSynchronizationAdapter {

        private final LoggerAndEventsHolder tbLogger;

        @Override
        public void suspend() {
            TransactionSynchronizationManager.unbindResourceIfPossible(resourceKey(tbLogger.logger));
        }

        @Override
        public void resume() {
            TransactionSynchronizationManager.bindResource(resourceKey(tbLogger.logger), tbLogger);
        }

        @Override
        public void afterCompletion(int status) {
            if (status == STATUS_COMMITTED) {
                tbLogger.loggedEventInfos.forEach(lei ->
                        doLogEvent(tbLogger.logger, lei.event, lei.printDefaultValues));
            }
            TransactionSynchronizationManager.unbindResourceIfPossible(resourceKey(tbLogger.logger));
        }
    }

    private static void doLogEvent(Logger logger, Message event, boolean printDefaultValues) {
        try {
            JsonFormat.Printer printer = JsonFormat.printer().omittingInsignificantWhitespace();
            if (printDefaultValues) {
                printer = printer.includingDefaultValueFields();
            }
            logger.info(printer.print(event));
        } catch (InvalidProtocolBufferException e) {
            log.error("Error logging transaction bound events", e);
        }
    }

    private static String resourceKey(Logger logger) {
        return AfterCommitEventLogging.class.getName() + "_" + logger.getName();
    }
}
