package ru.yandex.travel.workflow;

import java.time.Instant;
import java.util.Optional;
import java.util.function.Supplier;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;

import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.ClientCall;
import ru.yandex.travel.workflow.repository.ClientCallRepository;

@RequiredArgsConstructor
public class ClientCallService {

    public static final String TASK_KEY = "CLEANUP_CLIENT_CALLS";

    private final ClientCallRepository clientCallRepository;

    private final ClientCallServiceProperties clientCallServiceProperties;

    @TransactionMandatory
    @SuppressWarnings("unchecked")
    public <Rsp extends Message> Rsp computeIfAbsent(String clientCallId, Supplier<Rsp> responseSupplier) {
        Optional<ClientCall> maybeExistingCall = clientCallRepository.findById(clientCallId);
        if (maybeExistingCall.isPresent()) {
            return (Rsp) maybeExistingCall.get().getResponse();
        } else {
            Rsp result = responseSupplier.get();
            ClientCall call = new ClientCall();
            call.setId(clientCallId);
            call.setResponse(result);
            // we flush as this method will be usually called at the start of transaction before any data is read
            // so we can fail fast early
            clientCallRepository.saveAndFlush(call);
            return result;
        }
    }

    @TransactionMandatory
    public void cleanupClientCalls(String ignoredTaskKey) {
        clientCallRepository.cleanupClientCalls(
                Instant.now().minusMillis(clientCallServiceProperties.getTtl().toMillis())
        );
    }
}
