package ru.yandex.travel.workflow;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.RegularTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.workflow.repository.ClientCallRepository;

@Configuration
@EnableConfigurationProperties(ClientCallServiceProperties.class)
@ConditionalOnProperty("client-call-service.enabled")
@RequiredArgsConstructor
public class ClientServiceConfiguration {

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final ClientCallServiceProperties clientCallServiceProperties;

    @Bean
    public ClientCallService clientCallService(ClientCallRepository clientCallRepository) {
        return new ClientCallService(clientCallRepository, clientCallServiceProperties);
    }

    @Bean
    public TaskProcessor<String> clientCallsCleanupTaskProcessor(ClientCallService clientCallService) {
        TaskKeyProvider<String> taskKeyProvider = new RegularTaskKeyProvider<>(ClientCallService.TASK_KEY);

        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("ClientCallCleanup");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(
                taskKeyProvider,
                clientCallService::cleanupClientCalls,
                forcedRollbackTxManagerWrapper,
                txDefinition,
                TaskProcessorProperties.builder()
                        .name("ClientCallCleanupScheduler")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(1)
                        .initialStartDelay(clientCallServiceProperties.getCleanupStartDelay())
                        .scheduleRate(clientCallServiceProperties.getCleanupInterval())
                        .build()
        );
    }
}
